'use strict';
define(function() {
  /**
   * DualListBox Directive
   * Provides two listBox with drag&drop functionnality
   */

  var dualListGroup = function($filter, gaJsUtils, $timeout) {
    return {
      restrict: 'A',
      scope: {
        data: '=',
        addfilters: '=?',
        noSortLeft: '=?',
        noSortRight: '=?',
        afterMoveItem: '&?',
        afterMoveItems: '&?',
        // on-change: is called when data change
        // exemple-> on-change="myfunction(myparam)"
        onChange: '&?'
      },
      link: function(scope, elt) {
        // Instanciate both data lists as array if needed
        if (!scope.data) {
          scope.data = {};
        }
        if (!angular.isArray(scope.data.leftData)) {
          scope.data.leftData = [];
        }
        if (!angular.isArray(scope.data.rightData)) {
          scope.data.rightData = [];
        }
        if(!scope.filter){
          scope.filter = {};
        }

        // ajoute une classe tabs à la div principale du template html
        // permet d'appliquer du css spécifique aux dualListBox qui contiennent des tabs
        scope.hasTabs
          = Object.prototype.hasOwnProperty.call(
            scope.data, 'tabs') && typeof scope.data.tabs === 'string';

        function checkSourceAgainstSelected() {
          /*
                 If a source is provided,
                 the data from that side cannot contain data from the other side
                 ex: usersGroups-groups, groups is the source and cannot contain
                 data from userGroups
                 */
          if (angular.isDefined(scope.data.source)) {
            const source =
              scope.data.source === 'left'
                ? scope.data.leftData
                : scope.data.rightData;
            const destination =
              scope.data.source === 'left'
                ? scope.data.rightData
                : scope.data.leftData;

            destination.forEach(function(d) {
              source.forEach(function(s, i) {
                /*
                  TODO: Should be the right way!
                  TODO: Didnt work while comparing groups cause
                  TODO: users.groups has no linked roles, whereas scope.resources.groups does!
                  if(angular.equals(s, d)){
                      source.splice(i,1);
                  }*/
                if (s.uid == undefined || d.uid == undefined) {
                  if (angular.equals(s, d)) {
                    source.splice(i, 1);
                  }
                } else if (s.uid == d.uid) {
                  source.splice(i, 1);
                }
              });
            });
          }
        }

        /**
         * Drop Success Handler
         * Remove index from array
         *
         * @param $event
         * @param index rang de l'élement glissé/déposé dans le tableau d'origine
         * @param array tableau d'origine de l'élément glissé/déposé
         * @param fromSide côté d'origine de l'élément glissé/déposé
         *  (ex. 'left' pour un élément glissé de la gauche vers la droite)
         */
        scope.dropSuccessHandler = function($event, index, array) {

          if (!scope.data.disableDragFromRight) {
            array.splice(index, 1);
          }
          else if (!~scope.data.disableDragFromRight.indexOf(array[index][scope.data.keytocheck])) {
            array.splice(index, 1);
          }
          // après drop d'un élément, exécute une méthode du composant parent
          // si celui-ci doit agir sur cet évènement (ex. setRemoteAttribute)
          if (scope.afterMoveItem && typeof scope.afterMoveItem === 'function') {
            $timeout(scope.afterMoveItem());
          }
        };

        scope.moveItem = (array, index, direction) => {
          gaJsUtils.moveElementInArray(array, index, direction);
        };
        scope.isFirstItem = (index) => {
          return index === 0;
        };
        scope.isLastItem = (array, index) => {
          return index === array.length -1;
        };

        /**
         * Gère le dépôt d'un élément ($data) dans un tableau cible spécifié.
         *
         * @param {Event} $event - L'événement de dépôt déclenché par l'interaction utilisateur.
         * @param {Object} $data - L'objet de données qui est déposé.
         * @param {Array} array - Le tableau cible dans lequel les données doivent être ajoutées.
         * @param {boolean} fromLeft - Indique si les données proviennent de la liste de gauche.
         */
        scope.onDrop = function($event, $data, array, fromLeft) {
          $timeout(() => {
            // Détermine le filtre cible et vérifie s'il est actif (utilisé pour filtrer les données)
            const targetFilter = fromLeft ? scope.filter.rightFilter : scope.filter.leftFilter;
            const isTargetFiltered = fromLeft ? typeof targetFilter === 'string' && targetFilter.length > 0 : false;

            // S'assure que le tableau cible est bien un tableau ; l'initialise si ce n'est pas le cas
            if(!angular.isArray(array)){
              array = [];
            }

            // Si le déplacement depuis la liste de droite n'est pas désactivé :
            if (!scope.data.disableDragFromRight) {
              // Vérifie si l'objet $data n'existe pas déjà dans le tableau cible
              if (!array.some(el => el.name === $data.name)) {
                // Ajoute l'objet $data au tableau cible
                array.push($data);

                // Si un filtre est actif sur la cible, met à jour les données sauvegardées
                if (isTargetFiltered) {
                  updateBackupData($data, fromLeft);
                }
              }
            }else {
              // Si le déplacement depuis la liste de droite est désactivé pour cet élément spécifique :
              if (~scope.data.disableDragFromRight.indexOf($data[scope.data.keytocheck])) {
                // Affiche un avertissement si l'élément est interdit de déplacement
                require('toastr').warning(
                  $filter('translate')('hpo.model.calage.required')
                );
              } else {
                // Sinon, ajoute l'objet $data au tableau cible s'il n'existe pas déjà
                if (!array.some(el => el.name === $data.name)) {
                  array.push($data);
                  // Met à jour les données sauvegardées si un filtre est actif
                  if (isTargetFiltered) {
                    updateBackupData($data, fromLeft);
                  }
                }
              }
            }

            // Si on est pas dans le cas d'une fiche-objet avec onglets
            if (!gaJsUtils.notNullAndDefined(scope.data.tabs)) {
              $timeout(() => {
                // Supprime l'élément $data de la liste source (gauche ou droite)
                const sourceList = fromLeft ? scope.data.leftData : scope.data.rightData;
                const indexInSource = sourceList.findIndex(el => el.name === $data.name);
                if (indexInSource >= 0) {
                  sourceList.splice(indexInSource, 1);
                }
              });
            }

            // Trie la liste cible si le tri automatique est activé pour cette direction
            if((!fromLeft && !scope.noSortLeft) || (fromLeft && !scope.noSortRight)){
              gaJsUtils.sortDataByNameIgnoreCase(array);
            }
            if (typeof scope.onChange === 'function') {
              // call the function on next $apply to make
              // sure the ng-model has been updated
              $timeout(scope.onChange);
            }
          })
        };

        let setleftBackupData = () => {
          if(scope.data.tabs === 'left'){
            if(!scope.data.leftBackupData){
              scope.data.leftBackupData = [];
            }
            for(let i=0;i<scope.data.leftData.length;i++){
              scope.data.leftBackupData[i] = scope.data.leftData[i].fields;
            }
          }else{
            scope.data.leftBackupData = scope.data.leftData;
          }
        }

        /**
         * Met à jour les données de sauvegarde en déplaçant un élément d'un côté à l'autre.
         * La méthode identifie de quel côté (gauche ou droit) provient l'élément donné,
         * le retire de sa liste d'origine et l'ajoute à la liste cible opposée.
         *
         * @param {Object} droppedElement - L'élément à déplacer entre les listes de sauvegarde.
         * @param {Object} fromLeft - true quand l'élement provient de la liste de gauche
         */
        const updateBackupData = (droppedElement, fromLeft) => {

          // Détermine la liste source (d'où provient l'élément) et la liste cible (où il doit aller)
          const sourceSideList = fromLeft ? scope.data.leftBackupData : scope.data.rightBackupData;
          const targetSideList = fromLeft ? scope.data.rightBackupData : scope.data.leftBackupData;

          // Ajoute l'élément à la liste cible
          targetSideList.push(droppedElement);

          // Trouve l'index de l'élément dans la liste source
          const index = sourceSideList.findIndex(att => att.name === droppedElement.name);


          // Si l'élément existe dans la liste source, le supprime de cette liste
          if (index >= 0) {
            sourceSideList.splice(index, 1);
          }
          gaJsUtils.sortDataByNameIgnoreCase(targetSideList);
        };

        scope.$on('setDualListBoxContent', function(event, params) {
          if (params.id === scope.data.id) {
            if (params.leftData) {
              scope.data.leftData = params.leftData;
              setleftBackupData();
            }
            if (params.rightData) {
              scope.data.rightData = params.rightData;
              scope.data.rightBackupData = params.rightData;
            }
            checkSourceAgainstSelected();
            if (params.tellItIsDone)
              params.tellItIsDone.done = true;
          }
        });

        scope.$on('initFilter', () => {
          if(angular.isDefined(scope.filter.rightFilter)){
            scope.filter.rightFilter = "";
            scope.filterBox(scope.filter.rightFilter, false);
          }
          if(angular.isDefined(scope.filter.leftFilter)){
            scope.filter.leftFilter = "";
            scope.filterBox(scope.filter.leftFilter, true);
          }
        });

        checkSourceAgainstSelected();

        /**
         * En cas d'actualisation des données d'entrée de la dual-list-box,
         * prépare les données de backup pour faire fonctionner le filtre
         * et enlève les doublons si "removeDuplicates"
         */
        const onChangeData = () => {
          if(scope.data.removeDuplicates && scope.data.rightData && scope.data.leftData) {
            for (const leftElement of scope.data.leftData) {
              for (let i=scope.data.rightData.length-1; i>=0; i--) {
                if(leftElement[scope.data.leftDisplayAttribute] ===
                    scope.data.rightData[i][scope.data.rightDisplayAttribute]) {
                  scope.data.rightData.splice(i, 1);
                }
              }
            }
          }
          if (scope.addfilters) {
            scope.data.rightBackupData = scope.data.rightData;
            setleftBackupData();
          }
        };
        // Préparation des données à l'initialisation
        onChangeData();
        // Broadcast à utiliser lorsque l'on change les données de la directive déjà chargée
        scope.$on('changeDualListBoxData', onChangeData);

        /**
         * Filtre la liste de données (partie gauche ou droite) en fonction de la saisie dans un champ de filtre.
         * Si l'input de filtre est vide, réinitialise les données à leur état original.
         * @param {string} filterValue - Valeur saisie dans le champ de filtre
         * @param {boolean} isLeftData - Indique si le filtre s'applique à la partie gauche (true) ou droite (false)
         */
        scope.filterBox = (filterValue, isLeftData) => {
          if (isLeftData) { // Cas où le filtre s'applique à la partie gauche
            if(scope.data.tabs === 'left') {
              // Filtre chaque champ de 'leftData' individuellement si l'onglet actuel est 'left'
              for(let i=0; i<scope.data.leftData.length; i++) {
                scope.data.leftData[i].fields = filterList(scope.data.leftBackupData[i], filterValue, isLeftData, scope.data.rightData);
              }
            } else {
              // Filtre globalement la liste 'leftData' si les onglets ne portent pas sur la partie gauche ('left')
              scope.data.leftData = filterList(scope.data.leftBackupData, filterValue, isLeftData, scope.data.rightData);
            }
          } else {
            // Cas où le filtre s'applique à la partie droite
            const hasTabs = typeof scope.data.tabs === 'string' && scope.data.tabs.length > 0;
            const elementsToExclude = hasTabs ? scope.data.leftData.flatMap(tab => tab.fields) : scope.data.leftData;
            scope.data.rightData = filterList(scope.data.rightBackupData, filterValue, isLeftData, elementsToExclude);
          }
        };

        /**
         * Filtre la liste en entrée et renvoie une liste d'objets ayant une propriété 'name'
         * contenant la string filtervalue en paramètre
         * @param list liste d'objets à filtrer où chaque élément comporte une propriété 'name'
         * @param filterValue string qui doit être contenu dans la propriété 'name' de tous les éléments de la liste de sortie
         * @param isLeftData true quand filtervalue est saisi dans l'input du filtre de la partie gauche de la dualList
         * @param elementsToExclude objets déjà présents dans la liste opposée à ne pas proposer dans la liste filtrée
         * @return {[string]} objets ayant une propriété 'name' contenant la string filtervalue en paramètre
         */
        const filterList = (list, filterValue, isLeftData, elementsToExclude = []) => {
          let attribute;
          if (angular.isDefined(isLeftData)) {
            if (isLeftData) {
              attribute = scope.data.leftDisplayAttribute ? scope.data.leftDisplayAttribute : 'name';
            } else {
              attribute = scope.data.rightDisplayAttribute ? scope.data.rightDisplayAttribute : 'name';
            }
          } else {
            attribute = 'name';
          }
          const excluded = elementsToExclude.length > 0 ? elementsToExclude.map(el => el[attribute].toLowerCase()) : [];
          return list.filter(el => !excluded.includes(el[attribute].toLowerCase())
                  && (filterValue.length === 0 || (filterValue.length > 0
                      && el[attribute].toLowerCase().includes(filterValue.toLowerCase()))));
        };

        /**
         * Au clic sur le bouton flèche gauche/droite
         * déplace toutes les valeurs d'une liste à une autre.
         * @param fromLeft est true au clic sur le bouton de la flèche vers la droite
         */
        scope.moveAllItems = (fromLeft) => {
          const source = fromLeft ? 'leftData' : 'rightData';
          const target = fromLeft ? 'rightData' : 'leftData';

          if(scope.data.tabs === 'left'){
            if(fromLeft){
              scope.data[target].push(...scope.data[source][scope.data.activeTab].fields)
            }else{
              scope.data[target][scope.data.activeTab].fields.push(...scope.data[source]);
            }
          }else{
            scope.data[target].push(...scope.data[source]);
          }

          // au clic sur la flèche pour tout ajouter/enlever, envoie un évènement au composant parent si celui-ci doit agir sur cet évènement (ex. setRemoteAttribute)
          if (scope.afterMoveItems && typeof scope.afterMoveItems === 'function') {
            scope.afterMoveItems();
          }

          // pour ne pas perdre le passage en référence des variable il faut pas faire scope.data[source] = []
          if(scope.data.tabs === 'left' && fromLeft &&
            Array.isArray(scope.data[source][scope.data.activeTab].fields) &&
            scope.data[source][scope.data.activeTab].fields.length>0){
            scope.data[source][scope.data.activeTab].fields.length = 0;
          }else if(Array.isArray(scope.data[source]) && scope.data[source].length>0){
            scope.data[source].length = 0;
          }
          if (typeof scope.onChange === 'function') {
            // call the function on next $apply to make
            // sure the ng-model has been updated
            $timeout(scope.onChange);
          }
        };

        if(!scope.noSortLeft){
          gaJsUtils.sortDataByNameIgnoreCase(scope.data.leftData);
        }
        if(!scope.noSortRight){
          gaJsUtils.sortDataByNameIgnoreCase(scope.data.rightData);
        }

        /**
         * cette methode ajoute un onglet aux listBox gauche
         */
        scope.addTabsToBranch = () => {
          if (angular.isUndefined(scope.data.leftData))
            scope.data.leftData = [];
          scope.data.leftData.push({
            title: $filter('translate')(
              'tools.builder.form.tabs.default_title'
            ),
            fields: [],
          });
          setleftBackupData();
          if (scope.data.leftData.length === 1) {
            scope.data.activeTab = 0;
          }

          // surveille le nombre d'onglets pour ajuster la position des parties gauche/milieu/droite et la hauteur du conteneur
          // la hauteur d'une titre d'onglet est de 41px
          adjustDualListBoxMarginTop();
        };

        /**
         * Au clic sur le bouton "Cliquer pour éditer" ou sur le bouton "Valider" (coche),
         * cette methode change entre le mode edition et le mode classique (0 ou 1)
         * @param {number} mode
         */
        scope.toggleTabEditMode = (mode) => {
          scope.editMode = mode;
        };
        /**
         * Au clic sur le bouton supprimer un onglet (croix rouge),<ul><li>
         * supprime l'onglet de la partie gauche</li><li>
         * attend la prochaine frame et ajuste la position verticale des 3 parties et la hauteur du conteneur
         * @param {number} index rang de l'onglet dans le tableau d'onglets
         */
        scope.removeTab = (index) => {
          const tabFields = scope.data.leftData[index].fields;

          // KIS-3332: quand je supprime un onglet alors je dois retrouver les attributs de l’onglet dans la liste des attributs disponibles
          if (Array.isArray(tabFields) && tabFields.length > 0) {
            scope.data.rightData.push(...scope.data.leftData[index].fields);
            gaJsUtils.sortDataByNameIgnoreCase(scope.data.rightData);
            for (const field of tabFields) {
              if (!scope.data.rightBackupData.find(field => field.name === field.name)) {
                scope.data.rightBackupData.push(field);
              }
            }
            gaJsUtils.sortDataByNameIgnoreCase(scope.data.rightBackupData);
          }

          scope.data.leftData.splice(index, 1);

          // surveille le nombre d'onglets pour ajuster la position des parties gauche/milieu/droite et la hauteur du conteneur
          // la hauteur d'une titre d'onglet est de 41px
          adjustDualListBoxMarginTop();
        };
        scope.switchedTab = -1;

        /**
         * cette methode permet de modifier la position des onglets
         */
        scope.moveTab = (index, direction) => {
          if (
            (index === 0 && direction === 'left') ||
            (index === scope.data.leftData.length - 1 && direction === 'right')
          )
            return false;
          var newTabs = angular.copy(scope.data.leftData),
            newIndex = direction === 'left' ? index - 1 : index + 1;
          newTabs.splice(index, 0, newTabs.splice(newIndex, 1)[0]);
          scope.data.leftData = newTabs;
          scope.switchedTab = newIndex;

          // KIS-336: actualise la propriété margin-top
          adjustDualListBoxMarginTop();

          // KIS-3340: le déplacement d'un onglet doit garder le focus sur cet onglet
          $timeout(() => {
            scope.data.activeTab = newIndex;
          });
        };

        /**
         * Au clic sur le bouton "Ajouter un onglet" et au clic sur le bouton "Supprimer un onglet",
         * lorsque une ligne de titre d'onglets est ajoutée/supprimée:<ul><li>
         * modifie la position verticale des parties gauche et droite</li><li>
         * modifie la position verticale des flèches gauche/droite dans la partie du milieu</li><li>
         * modifie la hauteur du conteneur commun des parties gauche/milieu/droit
         * KIS-2987
         */
        const adjustDualListBoxMarginTop = () => {
          $timeout(() => {
            if (elt && elt[0]) {

              // conteneur des onglets
              const tabsContainer = elt[0].querySelector('.nav.nav-tabs');

              if (tabsContainer) {

                // méthode interne qui modifie la propriété marginBottom ou marginTop du style de l'élément
                const modifyElementStyle = (style, marginSide) => {
                  const marginValue = (scope.tabsLineCount -1) * tabLineHeight;
                  const marginProperty = 'margin' + marginSide[0].toUpperCase() + marginSide.substring(1);
                  style[marginProperty] = marginValue + 'px';
                };

                // évalue la hauteur du conteneur des onglets
                scope.tabsLineCount = Math.round(tabsContainer.clientHeight / tabLineHeight);

                // liste de la partie gauche de chaque onglet
                const leftListsContainer = tabsContainer.nextSibling;
                const leftLists = leftListsContainer.querySelectorAll('.list');

                if (leftLists) {
                  for (const leftList of leftLists) {
                    // modifie la position verticale de la liste de gauche (div absolute)
                    modifyElementStyle(leftList.style, 'top');
                  }
                }

                // conteneur des parties gauche et droite:
                const dualListContainer = tabsContainer.closest('.dualListBox');

                // agrandissement/rétrécissement de la hauteur
                if (dualListContainer) {
                  modifyElementStyle(dualListContainer.style, 'bottom');

                  // liste de la partie droite
                  // modifie la position verticale de la liste de droite (div absolute)
                  const rightList = dualListContainer.querySelector('.right-part .list');
                  if (rightList) {
                    modifyElementStyle(rightList.style, 'top');
                  }

                  // partie du milieu: flèches gauche/droite
                  // modifie la position verticale de la partie du milieu (div absolute)
                  const middlePart = dualListContainer.querySelector('.middle-part');
                  if (middlePart) {
                    modifyElementStyle(middlePart.style, 'top');
                  }

                  // filtres de texte
                  // modifie la position verticale des filtres
                  const filters = dualListContainer.querySelectorAll('.dualListBox-filters');
                  if (filters) {
                    for (const filterElement of filters) {
                      modifyElementStyle(filterElement.style, 'top');
                    }
                  }
                }
              }
            }
          });
        };

        /**
         * Evalue si la liste de la partie gauche et la liste de la partie droite sont situées à la même position verticale
         * @param {HTMLElement} tabsContainer conteneur des lignes de titres d'onglets
         * @return {boolean} true si la partie gauche et la liste de la partie droite sont situées à la même position verticale
         * KIS-2987
         */
        const leftAndRightPartHasSameVerticalPosition = (tabsContainer) => {
          const dualListContainer = tabsContainer.closest('.dualListBox');
          if (dualListContainer) {
            const leftList = dualListContainer.querySelector('.left-part .list');
            const rightList = dualListContainer.querySelector('.right-part .list');
            if (leftList && rightList) {
              return leftList.offsetTop === rightList.offsetTop;
            } else {
              // attend la prochaine frame pour l'affichage des listes
              $timeout(() => {
                leftAndRightPartHasSameVerticalPosition(tabsContainer);
              });
            }
          } else {
            $timeout(() => {
              // attend la prochaine frame pour l'affichage de la dualListBox
              leftAndRightPartHasSameVerticalPosition(tabsContainer);
            });
          }
        };

        /**
         * Effectue le décompte du nombre de lignes de titres d'onglets
         * Si la popup contient plus d'1 ligne d'onglets alors
         * on décale les listes des parties gauche et droite, la partie du milieu
         * et on agrandit la hauteur du conteneur de la dualistbox
         * @param {HTMLElement} tabsContainer conteneur des lignes de titres d'onglets
         * KIS-2987
         */
        const initDualListBoxOffset = (tabsContainer = null) => {
          if (scope.hasTabs && elt && elt[0]) {
            if (!tabsContainer) {
              tabsContainer = elt[0].querySelector('.nav.nav-tabs');
            }
            if (tabsContainer) {

              if (leftAndRightPartHasSameVerticalPosition(tabsContainer)) {

                // hauteur d'une ligne de titres d'onglets
                adjustDualListBoxMarginTop();
              } else {

                // attend que l'affichage de la popup tienne compte de la position absolute des div de la dualListBox
                $timeout(() => {
                  initDualListBoxOffset(tabsContainer);
                });
              }
            } else {
              // si la dualListBox n'est pas encore affiché on attend la frame suivante
              $timeout(() => {
                initDualListBoxOffset();
              });
            }
          }
        };

        // Hauteur d'une ligne d'onglets
        // base de calcul de la propriété margin-top des parties gauche/droite
        const tabLineHeight = 41;

        // Dès l'initialisation, effectue le décompte du nombre de lignes de titres d'onglets
        initDualListBoxOffset();

      },
      templateUrl: 'js/XG/modules/common/views/directives/dualListBox.html',
    };
  };

  dualListGroup.$inject = ['$filter', 'gaJsUtils', '$timeout'];
  return dualListGroup;
});
