'use strict';

/**
 * @ngdoc controller
 * @name modules.tools.controllers:EdigeoCtrl
 * @description
 * The EDIGEO controller
 */
define(function () {
  const EdigeoCtrl = function (
    $scope,
    ngDialog,
    $filter,
    FilesFactory,
    ImportExportFactory,
    DataStoreFactory,
    processFactory,
    ngTableParams,
    $interval,
    gaDomUtils,
    FeatureAttachmentFactory,
    $timeout,
    ParametersFactory,
    SridFactory,
    gaJsUtils,
    DecoupAdminFactory,
    ConfigFactory
  ) {
    const TYPE = 'load edigeo';

    $scope.dialogContent =
      '<div>Confirmez vous la suppresion de [!!1!!] ?</div>';
    $scope.dialogContent += '<div>';
    $scope.dialogContent +=
      "  <button ng-click='removeAttachedFile()'>Oui</button>";
    $scope.dialogContent +=
      "  <button ng-click='cancelRemoveAttachedFile()'>Non</button>";
    $scope.dialogContent += '</div>';

    $scope.initFile = (inputFile) => {
      $scope.lastFile = inputFile.files[0];
      //inputFile.value = null;
    };

    $scope.srids = SridFactory.sridsList;

    /**
     *    Chargement du fichier EDIGEO (fichier compressé) sur le serveur.
     *  Le chargement met le fichier en document attaché du layer "edigeo"
     *  pour l'identifiant qui est le timestamp du moment du chargement.
     */
    $scope.uploadEdigeoFile = () => {
      if ($scope.lastFile) {
        $scope.fileToUploadName = $scope.lastFile.name;
        $scope.wait = true;
        gaDomUtils.showGlobalLoader();
        const idfile = Date.now();

        FeatureAttachmentFactory.uploadFile($scope.lastFile, 'edigeo', idfile).then(
          () => {
            $scope.wait = false;
            $scope.lastFile = null;
            $timeout(() => {gaDomUtils.hideGlobalLoader();}, 750);
            $scope.getAlreadyUploadedZips();
          },
          () => {
            gaDomUtils.hideGlobalLoader();
            $scope.getAlreadyUploadedZips();
          }
        );
      }
    };

    /**
     * Récupération de la liste des départements français.
     * On essaie de consommer geo.api.gouv sinon csv
     * Ils sont mis dans la liste exploitée par la liste déroulante.
     */
    $scope.loadDepartements = () => {
      DecoupAdminFactory.getDepartements()
        .then((res) => {
          if (res.status === 200 && res.data.push) {
            $scope.departements = [];
            for (const item of res.data) {
              const departement = {
                departmentCode: null,
                departmentName: null,
              };
              if (item.code.substring(0, 1) === '0') {
                departement.departmentCode = item.code.slice(1);
                departement.departmentName =
                  departement.departmentCode + ' ' + item.nom;
              } else {
                departement.departmentCode = item.code;
                departement.departmentName = item.code + ' ' + item.nom;
              }
              $scope.departements.push(departement);
            }
          }
        })
        .catch((error) => {
          require('toastr').error(
            'Impossible de récupérer les départements sur https://api.geo.gouv.fr ' +
              error
          );
          $scope.getDepartementsInCsv();
        });
    };
    $scope.getDepartementsInCsv = () => {
      FilesFactory.getListFromCsv('[cad]', 'departements', true)
        .then((res) => {
          if (res.data.push) {
            $scope.departements = res.data;
            for (let iDep = 0; iDep < res.data.length; iDep++) {
              res.data[iDep].departmentName =
                  res.data[iDep].departmentCode +
                  ' ' +
                  res.data[iDep].departmentName;
            }
          }
        })
        .catch((err) => {
          let error;
          if(err.data && err.data.error) {
            error = err.data.error
          } else if (err.data && err.data.message) {
            error = err.data.message
          } else {
            error = err;
          }
          require('toastr').error(
            'Impossible de récupérer les communes du département depuis le CSV ' +
              error
          );
        });
    };

    /**
     *  Construction de la liste des noms des composants qui vont recevoir les données.
     *  Quand une année est choisie ou un texte a été saisi,
     *  on suffixe avec souligné suivi de l'année ou du texte.
     *  Si rien n'a tété saisie le nom sans aucun suffixe est affiché.
     *  les noms affichées correspondent aux noms des composants
     *  qui seront créés pour stocker les données.
     */
    $scope.componentListBuildDestinations = () => {
      let suffix;
      const cList = $scope.edigeoComponents.list;
      if ($scope.edigeoComponents.suffix === undefined || $scope.edigeoComponents.suffix === null ||
          $scope.edigeoComponents.suffix.length === 0 ) {
        suffix = '';
      } else {
        suffix = '_' + $scope.edigeoComponents.suffix;
      }
      for (let ind = 0; ind < cList.length; ind++) {
        cList[ind].destination = cList[ind].source + suffix;
      }
    };

    $scope.loadComponentList = () => {
      $scope.edigeoComponents = {};
      $scope.edigeoComponents.list = [
        { source: 'BATIMENT', toLoad: true },
        { source: 'BORNE', toLoad: true },
        { source: 'COMMUNE', toLoad: true },
        { source: 'LIEUDIT', toLoad: true },
        { source: 'NUMVOIE', toLoad: false },
        { source: 'PARCELLE', toLoad: true },
        { source: 'PARCELLE_txt', toLoad: false },
        { source: 'PTCANV', toLoad: true },
        { source: 'SECTION', toLoad: true },
        { source: 'SUBDFISC', toLoad: true },
        { source: 'SUBDSECT', toLoad: true },
        { source: 'SYMBLIN', toLoad: false },
        { source: 'TLINE', toLoad: false },
        { source: 'TPOINT', toLoad: false },
        { source: 'TRONFLUV', toLoad: false },
        { source: 'TRONROUTE', toLoad: false },
        { source: 'TSURF', toLoad: false },
        { source: 'VOIEP', toLoad: true },
        { source: 'ZONCOMMUNI', toLoad: false },
      ];
      $scope.componentListBuildDestinations();
    };

    /**
     * Récupération de a liste des communes du département choisi
     * pour alimenter la liste déroulante du choix de la commune.
     * On essaie de consommer geo.api.gouv sinon csv
     */
    $scope.getCommunesOfDept = () => {
      if (
        $scope.departement !== undefined &&
        $scope.departement.choice !== undefined
      ) {
        //appel api sinon récupère csv
        let departmentCode = '0';
        if ($scope.departement.choice.length === 1) {
          departmentCode = departmentCode.concat($scope.departement.choice);
        } else {
          departmentCode = $scope.departement.choice;
        }
        DecoupAdminFactory.getCommunesByDepartement(departmentCode)
          .then((res) => {
            if (res.status === 200 && res.data.push) {
              $scope.communes = [];
              for (const item of res.data) {
                const commune = {
                  Code_commune_INSEE: null,
                  Nom_commune: null,
                };
                commune.Code_commune_INSEE = item.code;
                commune.Nom_commune = item.nom.toUpperCase();
                $scope.communes.push(commune);
              }
            }
          })
          .catch((error) => {
            require('toastr').error(
              'Impossible de récupérer les communes du département ' +
                $scope.departement.choice +
                ' sur https://api.geo.gouv.fr ' +
                error
            );
            $scope.getCommunesOfDeptInCsv($scope.departement.choice);
          });
      }
    };

    $scope.getCommunesOfDeptInCsv = () => {
      const query = 'Code_commune_INSEE m3startsWith ' + $scope.departement.choice;
      FilesFactory.getListFromCsv('[cad]', 'laposte_hexasmal', true, query)
        .then((res) => {
          if (res.data.push) {
            $scope.communes = res.data;
          }
        })
        .catch(() => {
          require('toastr').error(
            'Impossible de récupérer les communes du département ' +
              $scope.departement.choice +
              ' depuis le CSV '
          );
        });
    };

    $scope.loadDataStores = () => {
      DataStoreFactory.get().then(
          (res) => {
            $scope.dbselector = {};
            if (res.data.length !== 0) {
              $scope.datasources = res.data;
              if (res.data.length === 1) {
                $scope.dbselector.choice = res.data[0].name;
              } else {
                $scope.dbselector.choice = '';
              }
            }
          });
    };

    $scope.prepareLoadingSteps = () => {
      $scope.loadingSteps = [];
      $scope.loadingSteps.push({
        label: $filter('translate')('tools.majicedigeo.edigeo.recupfeuilles'),
        statusLbl: $filter('translate')('tools.majicedigeo.edigeo.waiting'),
        status: 'waiting',
      });
      $scope.loadingSteps.push({
        label: $filter('translate')(
          'tools.majicedigeo.edigeo.decompressionfeuilles'
        ),
        statusLbl: $filter('translate')('tools.majicedigeo.edigeo.waiting'),
        status: 'waiting',
      });
      $scope.loadingSteps.push({
        label: $filter('translate')('tools.majicedigeo.edigeo.conversionEnShp'),
        statusLbl: $filter('translate')('tools.majicedigeo.edigeo.waiting'),
        status: 'waiting',
      });
      $scope.loadingSteps.push({
        label: $filter('translate')('tools.majicedigeo.edigeo.chargementBDD'),
        statusLbl: $filter('translate')('tools.majicedigeo.edigeo.waiting'),
        status: 'waiting',
      });
      $scope.loadingLaunched = false;
    };

    $scope.getAlreadyUploadedZips = () => {
      if ($scope.edigeoFiles === undefined) $scope.edigeoFiles = [];
      ImportExportFactory.getUploadedEdigeo().then(
          (res) => {
            let dd;
            $scope.edigeoFiles.splice(0, $scope.edigeoFiles.length);
            if (res.data.length !== undefined) {
              for (let ii = 0; ii < res.data.length; ii++) {
                $scope.edigeoFiles.push(res.data[ii]);
                dd = res.data[ii].date;
                res.data[ii].attachmentObjectId = dd;
                res.data[ii].date = $scope.getFmtDate(new Date(dd));
              }
            }
          });
    };

    $scope.initListOfCommunes = (whichOne) => {
      let confname;
      if (whichOne === 'favorite'){
        //-- L'opérateur demande le chargement de la liste favorite
        confname = 'edigeo_liste_de_communes_favorite';
      } else {
        //-- A l'initialisation, on charge la liste précédemment construite.
        //-- Et on regarde si une liste favorite de communes existe.
        confname = 'edigeo_derniere_liste_de_communes';
        ParametersFactory.getbyname('edigeo_liste_de_communes_favorite').then(
          (res) => {
            if (res.data !== undefined && res.data !== '') {
              $scope.favoriteExists = true;
              $scope.edigeoCommuneFavDbparamId = res.data.id;
              $scope.dbParamCommuneFavList = res.data;
            }
          }
        );
      }
      $scope.dbParamCommuneList = {};
      ParametersFactory.getbyname(confname).then(
          (res) => {
            if (res.data !== undefined && res.data !== '') {
              if (whichOne === 'favorite') {
                $scope.edigeoCommuneFavDbparamId = res.data.id;
                $scope.dbParamCommuneFavList = res.data;
              } else {
                $scope.edigeoCommuneDbparamId = res.data.id;
                $scope.dbParamCommuneList = res.data;
              }
              if (res.data.data && res.data.data.list.length !== 0) {
                $scope.edigeoCommune.list.splice(
                    0,
                    $scope.edigeoCommune.list.length
                );
                $scope.departement.choice = res.data.data.list[0].departement;
                for (let ind = 0; ind < res.data.data.list.length; ind++) {
                  $scope.edigeoCommune.list.push(res.data.data.list[ind]);
                }
              }
            }
          });
    };

    $scope.getExistingProcesses = () => {
      let ts = 0;
      let indOfLast = -1;
      processFactory.getProcessByUsertype(TYPE).then(
          (res) => {
            if (res.data !== undefined && res.data.length !== 0) {
              for (let ind = 0; ind < res.data.length; ind++) {
                if (res.data[ind].etat === 'RUNNING') {
                  $scope.conversionIsRunning = $scope.loadingLaunched = true;
                  if (res.data[ind].both !== undefined) {
                    delete res.data[ind].both;
                  }
                  $scope.checkProcessInProgress(res.data[ind]);
                  const d1 = new Date(res.data[ind].creation);
                  $scope.readInfoFromExtraInfo(res.data[ind].extraInfos[0], d1);
                } else {
                  if (res.data[ind].creation > ts) {
                    ts = res.data[ind].creation;
                    indOfLast = ind;
                  }
                }
              }
              if (!$scope.conversionIsRunning && indOfLast !== -1) {
                $scope.conversionStopped(res.data[indOfLast]);
              }
            }
          });
    };

    $scope.initEventForVisibilityChange = () => {
      let visibilityChange;
      if (typeof document.hidden !== 'undefined') {
        // Opera 12.10 and Firefox 18 and later support
        $scope.hiddenEventName = 'hidden';
        visibilityChange = 'visibilitychange';
      } else if (document.hasOwnProperty('msHidden') && typeof document.msHidden !== 'undefined') {
        $scope.hiddenEventName = 'msHidden';
        visibilityChange = 'msvisibilitychange';
      } else if (document.hasOwnProperty('webkitHidden') && typeof document.webkitHidden !== 'undefined') {
        $scope.hiddenEventName = 'webkitHidden';
        visibilityChange = 'webkitvisibilitychange';
      }

      //-- Chrome et FF supportent l'événement visibilityChange mais pas forcément les autres.
      if (
        typeof document.addEventListener != 'undefined' &&
        typeof document.hidden != 'undefined'
      )
        document.addEventListener(
          visibilityChange,
          $scope.handleVisibilityChange,
          false
        );
    };

    $scope.init = () => {
      $scope.radEdigeoSource = {};
      $scope.radEdigeoSource.choice = 'file';
      $scope.replace = '1';
      $scope.edigeoCommune = {};
      $scope.edigeoCommune.list = [];
      $scope.conversionIsRunning = false;
      $scope.commune = {};
      $scope.departement = {};

      $scope.loadDataStores();
      $scope.loadComponentList();
      $scope.loadDepartements();

      /**
       * Au changement du select d'années du suffixe (mode === 'list')
       * Le changement du suffixe textuel est vérifié au blur
       * car la vérification ne doit pas être faite à chaque caractère modifié
       */
      $scope.$watch('edigeoComponents.suffix', () => {
            if ($scope.edigeoComponents.suffix !== undefined && $scope.mode === 'list') {
              // si le suffixe des années a été modifié
              const isYearSelected = $scope.edigeoComponents.suffix !== null && String(
                  $scope.edigeoComponents.suffix).length > 0;
              const isSuffixReset = $scope.componentHasSuffix && ($scope.edigeoComponents.suffix === null
                  || $scope.edigeoComponents.suffix === '');
              if (isYearSelected || isSuffixReset) {
                // si le suffixe est une année sélectionnée
                $scope.openWarnOrResetSuffix();
              }
            }
          }
      );
      $scope.$watch('departement.choice', $scope.getCommunesOfDept);

      $scope.prepareLoadingSteps();
      $scope.getAlreadyUploadedZips();
      $scope.initListOfCommunes();
      $scope.getExistingProcesses();
      $scope.initEventForVisibilityChange();
    };

    $scope.init();

    /**
     *     L'opérateur vient de cliquer sur la commande "+".
     *  On ajoute donc les communes sélectionnées dans la liste des communes
     *  pour lesquelles il faut importer les données EDIGEO.
     */
    $scope.addCommunes = () => {
      let ind2;
      let insee;

      //-- Parcours de la liste des communes choisies et donc à ajouter.
      for (let ind = 0; ind < $scope.commune.choice.length; ind++) {
        insee = $scope.commune.choice[ind].Code_commune_INSEE;
        if (insee.length === 4) {
          insee = '0' + insee;
        }
        //-- Rechecrche de la commune dans la liste des commune déjà choisies.
        for (ind2 = 0; ind2 < $scope.edigeoCommune.list.length; ind2++) {
          if ($scope.edigeoCommune.list[ind2].insee === insee) break;
        }

        //-- Si la commune n'est pas déja dans la liste, on l'ajoute.
        if (ind2 === $scope.edigeoCommune.list.length) {
          $scope.edigeoCommune.list.push({
            departement: $scope.departement.choice,
            nom: $scope.commune.choice[ind].Nom_commune,
            insee: insee,
          });
        }
      }
      $scope.saveListOfCommunes();
    };

    $scope.clearSelectedCommunesOK = () => {
      $scope.edigeoCommune.list.splice(0, $scope.edigeoCommune.list.length);
      ngDialog.closeAll();
    };

    $scope.clearSelectedCommunesCancel = () => {
      ngDialog.closeAll();
    };

    $scope.clearSelectedCommunes = () => {
      let dialogContent =
        '<div>Confirmez vous la suppression de toutes les communes choisies ?</div>';
      dialogContent += '<div>';
      dialogContent +=
        "  <button ng-click='clearSelectedCommunesOK()'>Oui</button>";
      dialogContent +=
        "  <button ng-click='clearSelectedCommunesCancel()'>Non</button>";
      dialogContent += '</div>';
      ngDialog.open({
        template: dialogContent,
        plain: true,
        scope: $scope,
      });
    };

    $scope.saveSelectedCommunesAsFavoriteOk = () => {
      $scope.saveListOfCommunes('favorite');
      ngDialog.closeAll();
    };

    $scope.saveSelectedCommunesAsFavorite = () => {
      let dialogContent;
      if ($scope.favoriteExists) {
        dialogContent =
          '<div style="max-width: 300px;">Confirmez vous le remplacement de la liste de commune favorite actuelle par celle présente dans la liste des communes sélectionées ?<br></div>';
        dialogContent += '<div>';
        dialogContent +=
          '  <button ng-click="saveSelectedCommunesAsFavoriteOk()">Oui</button>';
        dialogContent +=
          '  <button ng-click="clearSelectedCommunesCancel()">Non</button>';
        dialogContent += '</div>';
        ngDialog.open({
          template: dialogContent,
          plain: true,
          scope: $scope,
        });
      } else {
        $scope.saveListOfCommunes('favorite');
      }
    };

    $scope.removeCommune = (comInsee) => {

      //-- Rechecrche de la commune à supprimerdans la liste des communes choisies.
      for (let ind2 = 0; ind2 < $scope.edigeoCommune.list.length; ind2++) {

        if ($scope.edigeoCommune.list[ind2].insee === comInsee) {

          //-- La commune est trouvée, on l'enlève de cette liste des commaunes
          //-- dont on veut importer les données EDIGEO.
          $scope.edigeoCommune.list.splice(ind2, 1);
          break;
        }
      }
      $scope.saveListOfCommunes();
    };

    $scope.getResponseFile = (d) => {
      switch (d.etat) {
        case 'FINISHED':
          //$scope.refreshList();
          require('toastr').success('Chargement terminé avec succés');
          break;
        case 'FAILED':
          require('toastr').error(d.reason);
          break;
      }
    };

    $scope.setProgressBar = (value) => {
      const elt = document.getElementById('edigeo_pb');
      let iStart = 15;
      if ($scope.pbValuePrec === value) $scope.pbSameValueSince++;
      else $scope.pbSameValueSince = 0;
      $scope.pbValuePrec = value;
      elt.style.width = '' + value + '%';
      $scope.pbValue = '' + value + '%';
      if (++$scope.pbSameValueSince >= iStart && $scope.pbSameValueSince <= iStart + 2) {
        elt.style.backgroundColor = '#45f';
      } else if ( $scope.pbSameValueSince >= iStart + 3 && $scope.pbSameValueSince <= iStart + 4) {
        elt.style.backgroundColor = '#67f';
      } else if ($scope.pbSameValueSince >= iStart + 5 && $scope.pbSameValueSince <= iStart + 6) {
        elt.style.backgroundColor = '#45f';
      }else if ($scope.pbSameValueSince > iStart + 6) {
        elt.style.backgroundColor = '#67f';
        $scope.pbSameValueSince = 0;
      }
    };

    $scope.setStep = (data, finishedOrFailed) => {
      let iLastStepDone;
      $scope.setProgressBar(data.progress);
      if (data.extraInfo === null || data.extraInfo === undefined || data.extraInfo
          === 'recupfeuilles' || data.extraInfo === 'reinitialiseLot') {
        iLastStepDone = -1;
      } else if (data.extraInfo === 'decompressionfeuilles') {
        iLastStepDone = 0;
      }
      else if (data.extraInfo === 'conversionEnShp') {
        iLastStepDone = 1;
      }
      else if (data.extraInfo === 'chargementBDD') {
        iLastStepDone = 2;
      }

      if (finishedOrFailed === 'launching') {
        for (let i = 0; i < $scope.loadingSteps.length; i++) {
          $scope.loadingSteps[i].status = 'waiting';
          $scope.loadingSteps[i].statusLbl = $filter('translate')(
            'tools.majicedigeo.edigeo.waiting'
          );
          $scope.loadingSteps[i].errorMessage = '';
        }
      } else if (finishedOrFailed === 'finished') {
        for (let i = 0; i < $scope.loadingSteps.length; i++) {
          $scope.loadingSteps[i].status = 'done';
          $scope.loadingSteps[i].statusLbl = $filter('translate')(
            'tools.majicedigeo.edigeo.done'
          );
        }
      } else if (finishedOrFailed === 'failed') {
        $scope.loadingSteps[iLastStepDone + 1].status = 'failed';
        $scope.loadingSteps[iLastStepDone + 1].statusLbl = $filter('translate')(
          'tools.majicedigeo.edigeo.failed'
        );
        $scope.loadingSteps[iLastStepDone + 1].errorMessage = data.errorMessage;
      } else {
        for (let i = 0; i <= iLastStepDone; i++) {
          $scope.loadingSteps[i].status = 'done';
          $scope.loadingSteps[i].statusLbl = $filter('translate')(
            'tools.majicedigeo.edigeo.done'
          );
        }
        $scope.loadingSteps[iLastStepDone + 1].status = 'running';
        $scope.loadingSteps[iLastStepDone + 1].statusLbl = $filter('translate')(
          'tools.majicedigeo.edigeo.running'
        );
      }
      if (data.extraInfo === 'reinitialiseLot'){
        for (let index = 0; index < $scope.loadingSteps.length; index++) {
          $scope.loadingSteps[index].status = 'waiting';
          $scope.loadingSteps[index].statusLbl = $filter('translate')(
              'tools.majicedigeo.edigeo.waiting'
          );
          $scope.loadingSteps[index].errorMessage = '';
        }
      }
    };

    /**
     *     Suppression de tous les fichiers EDIGEO chargés sur le serveur.
     *  Cette fonction est appelée quand la conversion s'est terminée avec succés.
     */
    $scope.cleanEdigeoFiles = () => {
      let fileCnt = 0;

      if ($scope.radEdigeoSource.choice === 'file') {

        // Ce serait mieux de créer un tableau de promises et d'utiliser $q.all()

        //-- Dans le cas où on a chargé les données depuis un ZIP EDIGEO,
        //-- on supprime un à un chaque fichier EDIGEO (ZIP) chargé sur le serveur.
        for (let iFile = 0; iFile < $scope.edigeoFiles.length; iFile++) {
          FeatureAttachmentFactory.remove(
            $scope.edigeoFiles[iFile].name,
            'edigeo',
            $scope.edigeoFiles[iFile].attachmentObjectId
          ).then(() => {
            //-- Quand on vient de supprimer le dernier ficheir, on rafraichit la liste.
            if (++fileCnt >= $scope.edigeoFiles.length) {
              $scope.getAlreadyUploadedZips();
            }
          });
        }
      }
    };

    /**
     *     La conversion est finie, on affiche l'état de ce dernier traitement.
     */

    $scope.conversionStopped = (resdata) => {
     let extraInfo;

      $scope.conversionIsRunning = false;
      const creationDate = new Date(resdata.creation);
      const endDate = new Date(resdata.end);
      const etat = resdata.etat;
      if (resdata.hasOwnProperty('extraInfos') && resdata.extraInfos != null) {
        extraInfo = resdata.extraInfos[0];
      } else {
        extraInfo = '';
      }
      $scope.readInfoFromExtraInfo(
        extraInfo,
        creationDate,
        endDate,
        etat,
        resdata.errorMessage,
        resdata.warningMessage,
        false
      );
    };

    /**
     *     Gestion du  cas où la chargement des EDIGEOs a échoué.
     */
    const loadingEdigeoFailed = (res, repeat) => {
      if (res.data.errorMessage != null && res.data.errorMessage.indexOf('Use -s_srs to set one') !== -1) {
        $scope.edigeoSridNeeded = true;
        $scope.edigeoSridValue = undefined;
      }
      $interval.cancel(repeat);
      require('toastr').error($filter('translate')('tools.majicedigeo.edigeo.loadfailed'));
      $scope.setStep(res.data, 'failed');
      $scope.conversionStopped(res.data);
    };

    /**
     *     Fonction qui appel selon un interval de temps adaptif
     *  le service qui consulte l'état d'avancement de l'intégration.
     *
     *  resdata0: description du processus lancé.
     *  intervalSec:  Interval d'appel du service en MilliSecondes.
     */
    let stop;
    $scope.checkProcessInProgress = (resdata0, intervalMsec) => {
      if (intervalMsec === undefined) {
        intervalMsec = 750;
      }
      $scope.intervalCounter = 0;
      if (resdata0.creation)
        resdata0.creation = $filter('date')(
          new Date(resdata0.creation),
          'dd MMM yyyy HH:MM'
        );
      $scope.setStep(resdata0);
      stop = $interval(() => {
        if (resdata0.creation) {
          resdata0.creation = new Date(resdata0.creation).getTime();
        }
        if (resdata0.end) {
          resdata0.end = new Date(resdata0.end).getTime();
        }
        ImportExportFactory.getProgressionEdigeo(resdata0).then(
            (res) => {
              $scope.edigeoSridNeeded = false;
              if (res.data.etat === 'FINISHED' && res.data.progress === 100) {
                $interval.cancel(stop);
                $scope.getResponseFile(res.data);
                $scope.setStep(res.data, 'finished');
                $scope.cleanEdigeoFiles();
                //-- Un petit délai pour laisser le 100% de la barre de progression visible un instant.
                $timeout(() => {$scope.conversionStopped(res.data);}, 4000);
              } else if ($scope.loadingLaunched && res.data.etat === 'FAILED') {
                loadingEdigeoFailed(res, stop);
              } else if ($scope.loadingLaunched && res.data.etat === 'INTERRUPTED') {
                loadingEdigeoInterrupted(res, stop);
              } else {
                if (res.data.hasOwnProperty('extraInfo') && res.data.extraInfo === 'reinitialiseLot') {
                  $scope.setStep({progress: res.data.progress}, 'launching');
                }
                //-- Toujours en train de tourner
                $scope.setStep(res.data);
                //-- Indique le nombre de communes importées (traitement par lot)
                $scope.setCommunesLoadedNumber(res.data);
                // index du lot et nombre de communes dans le lot
                if (res.data.hasOwnProperty('extraInfos') && Array.isArray(res.data.extraInfos)
                    && res.data.extraInfos.length > 1) {
                  $scope.batchIndex = res.data.extraInfos[1];
                  if (res.data.extraInfos[2]) {
                    $scope.countBatchCommunes = res.data.extraInfos[2];
                  }
                }
                if ($scope.intervalCounter > 100 && intervalMsec < 90000) {
                  //-- Au plus c'est long, au moins on va interroger
                  //-- le serveur fréquemment sur l'état d'avancement.
                  $interval.cancel(stop);
                  $scope.checkProcessProgression(resdata0, 2 * intervalMsec);
                  $scope.intervalCounter = 0;
                }
              }
            },
            () => {
              if ($scope.loadingLaunched) {
                gaDomUtils.removeLocalLoader('.' + $scope.widgetName);
                $interval.cancel(stop);
                require('toastr').error(
                    $filter('translate')('tools.majicedigeo.edigeo.loadfailed')
                );
                $scope.conversionIsRunning = false;
              }
            }
        );
      }, intervalMsec);
    };

    /**
     * Affichage du nombre de communes importées durant le traitement par lot
     */
    $scope.setCommunesLoadedNumber = (data) => {
      if (data.hasOwnProperty('edigeoCommunesLoaded') && data.edigeoCommunesLoaded > 0) {
        $scope.communesBatchImported = parseInt(data.edigeoCommunesLoaded);
      }
    };

    $scope.loadEdigeoIntoDb = () => {
      let srid;
      let communes = '';
      let typeope = '';
      $scope.loadingLaunched = true;
      $scope.setStep({ progress: 0 }, 'launching');
      $scope.conversionIsRunning = true;
      $scope.setCurrentLoadDescription(new Date());

      if ($scope.radEdigeoSource.choice === 'web') {
        for (let i = 0; i < $scope.edigeoCommune.list.length; i++) {
          if (i !== 0) {
            communes += ',';
          }
          communes += $scope.edigeoCommune.list[i].insee;
        }
      }
      if ($scope.replace === '1') typeope = 'replace';

      if ($scope.edigeoSridNeeded || $scope.departement.choice > 100)
        srid = $scope.edigeoSridValue;

      const suffix = gaJsUtils.notNullAndDefined($scope.edigeoComponents.suffix) ? $scope.edigeoComponents.suffix : '';
      /*
       * http://127.0.0.1/services/kisf627b691-f98c-4f3a-8caf-7fe9c1b89611/
       *        loading/loadEdigeo?communes=13016&typedonneesentree=web&datasource=edg_tests
       *
       *    Le code département est nécessaire dans le cas ou des fichiers EDIGEO sont uploadés
       *  car cette information n'est pas disponible dans les fichiers fournis.'
       */
      ImportExportFactory.loadEdigeo(
        $scope.radEdigeoSource.choice,
        $scope.dbselector.choice,
        $scope.departement.choice,
        communes,
        $scope.transferid,
        suffix,
        $scope.edigeoComponents.list,
        typeope,
        srid
      ).then((res) => {
        if (res.data.etat !== 'RUNNING') {
          require('toastr').error(res.data.reason);
          $scope.conversionIsRunning = false;
          $scope.setStep(res.data, 'failed');
          $scope.conversionStopped(res.data);
        } else $scope.checkProcessInProgress(res.data);
      });
    };

    $scope.setCurrentLoadDescription = (datedebut) => {
      const nCommunes = $scope.edigeoCommune.list.length;

      if ($scope.radEdigeoSource.choice === 'web') {
        $scope.loadDescription =
          'Chargement des données cadastrales de ' + nCommunes + ' communes (';
        for (let ind = 0; ind < nCommunes && ind < 6; ind++) {
          if (ind !== 0) {
            $scope.loadDescription += ', ';
          }
          $scope.loadDescription += $scope.edigeoCommune.list[ind].insee;
        }
        if (nCommunes > 5) {
          $scope.loadDescription += ', ...';
        }
        $scope.loadDescription += ')';
      } else{
        $scope.loadDescription ='Chargement des données cadastrales du fichier ZIP ';
      }

      if ($scope.replace === '1') {
        $scope.loadDescription += ' en mode remplacement ';
      } else {
        $scope.loadDescription += ' en mode ajout ';
      }
      $scope.loadDescription += ' dans la base de données ' + $scope.dbselector.choice;
      $scope.loadDescription2 = 'Commencé le ' + datedebut.toLocaleDateString() +
        ' à ' + datedebut.toLocaleTimeString();
    };

    $scope.setPrevLoadDescription = (datedebut, datefin, etat, communes, errmess, warnings) => {

      const lst = communes.split(',');
      const nCommunes = lst.length;

      if (etat === 'FAILED') {
        $scope.previousLoadDescription = 'Echec du dernier chargement.  ';
      } else {
        $scope.previousLoadDescription = 'Succés du dernier chargement.  ';
      }
      if ($scope.radEdigeoSource.choice === 'web') {
        $scope.previousLoadDescription += 'Chargement précédent: données cadastrales de ' +
          nCommunes + ' communes (';

        for (let i = 0; i < nCommunes && i < 6; i++) {
          if (i !== 0) {
            $scope.previousLoadDescription += ', ';
          }
          $scope.previousLoadDescription += lst[i];
        }
        if (nCommunes > 5) $scope.previousLoadDescription += ', ...';
        $scope.previousLoadDescription += ')';
      } else {
        $scope.previousLoadDescription += 'Chargement précédent : données cadastrales du fichier ZIP ';
      }
      if ($scope.replace === '1') {
        $scope.previousLoadDescription += ' en mode remplacement ';
      } else {
        $scope.previousLoadDescription += ' en mode ajout ';
      }
      $scope.previousLoadDescription += ' dans la base de données ' + $scope.dbselector.choice;
      $scope.previousLoadDescription2 = 'Commencé le ' + datedebut.toLocaleDateString() +
        ' à ' + datedebut.toLocaleTimeString();
      $scope.previousLoadDescription3 = 'Terminé le ' + datefin.toLocaleDateString() +
        ' à ' + datefin.toLocaleTimeString();

      if (etat === 'FAILED') {
        $scope.previousLoadDescription4 = 'Erreur: ' + errmess;
      } else {
        $scope.previousLoadDescription4 = '';
      }
      if (warnings) {
        $scope.previousLoadWarnings1 = warnings;
      }
    };

    $scope.readInfoFromExtraInfo = (extraInfo, datedebut, datefin, etat, errmess, warnings, setSuffix) => {
      let currentLoad;
      if (extraInfo !== '') {
        currentLoad = JSON.parse(
          extraInfo
            .replace(/'/g, '"')
            .replace(/\\\\/g, '\\')
            .replace(/\\/g, '/')
        );
        $scope.radEdigeoSource.choice = currentLoad.typedonneesentree;
        if (currentLoad.hasOwnProperty('typeoperation') && currentLoad.typeoperation === 'replace') {
          $scope.replace = '1';
        }
        else {
          $scope.replace = '0';
        }
        $scope.dbselector.choice = currentLoad.datasource;
        if (setSuffix && gaJsUtils.isNumeric(currentLoad.suffix)) {
          $scope.edigeoComponents.suffix = parseInt(currentLoad.suffix);
        }
      }
      if (datefin === undefined) {
        //-- Pas de date de fin donc le process tourne toujours.
        $scope.setCurrentLoadDescription(datedebut);

      } else if (currentLoad !== undefined) {
        //-- Date de fin fournie donc le process est terminé.
        $scope.setPrevLoadDescription(
            datedebut,
            datefin,
            etat,
            currentLoad.communes,
            errmess,
            warnings
        );
      }
    };

    $scope.getFmtValPos2 = (val) => {
      if (val.length === 1) val = '0' + val;
      return val;
    };

    $scope.getFmtDate = (dd) => {
      let fmtDate = $scope.getFmtValPos2('' + dd.getDate()) + '/';
      fmtDate += $scope.getFmtValPos2('' + (1 + dd.getMonth())) + '/';
      fmtDate += dd.getFullYear() + ' ';
      fmtDate += $scope.getFmtValPos2('' + dd.getHours()) + ':';
      fmtDate += $scope.getFmtValPos2('' + dd.getMinutes()) + ':';
      fmtDate += $scope.getFmtValPos2('' + dd.getSeconds());

      return fmtDate;
    };

    $scope.removeAttachedFile = () => {
      FeatureAttachmentFactory.remove(
        $scope.fileToRemove.name,
        'edigeo',
        $scope.fileToRemove.attachmentObjectId
      ).then(() => {
        $scope.getAlreadyUploadedZips();
      });
      ngDialog.closeAll();
    };

    $scope.cancelRemoveAttachedFile = () => {
      ngDialog.closeAll();
    };

    $scope.removeUploadedFile = (aFile) => {
      $scope.fileToRemove = aFile;
      const reg = new RegExp('!!1!!', 'g');
      const template = $scope.dialogContent.replace(reg, aFile.name + ' du ' + aFile.date);
      ngDialog.open({ template: template, plain: true, scope: $scope });
    };

    $scope.saveListOfCommunes = (whichOne) => {
      let pid;
      let confname;
      if (whichOne === 'favorite') {
        pid = $scope.edigeoCommuneFavDbparamId;
        confname = 'edigeo_liste_de_communes_favorite';
      } else {
        pid = $scope.edigeoCommuneDbparamId;
        confname = 'edigeo_derniere_liste_de_communes';
      }

      if (pid === undefined) {
        ParametersFactory.add($scope.edigeoCommune, confname, confname).then(
          (res) => {
            if (res.data !== undefined) {
              if (whichOne === 'favorite') {
                $scope.edigeoCommuneFavDbparamId = res.data.id;
              } else {
                $scope.edigeoCommuneDbparamId = res.data.id;
              }
            }
          }
        );
      } else {
        if (whichOne === 'favorite') {
          if (!$scope.dbParamCommuneFavList) {
            $scope.dbParamCommuneFavList = {};
          }
          $scope.dbParamCommuneFavList.id = pid;
          if (!$scope.dbParamCommuneFavList.data) {
            $scope.dbParamCommuneFavList.data = {};
          }
          $scope.dbParamCommuneFavList.data.list = $scope.edigeoCommune.list;
          ParametersFactory.update(
            $scope.dbParamCommuneFavList,
            $scope.edigeoCommuneFavDbparamId
          );
        } else {
          $scope.dbParamCommuneList.data.list = $scope.edigeoCommune.list;
          ParametersFactory.update(
            $scope.dbParamCommuneList,
            $scope.edigeoCommuneDbparamId
          );
        }
      }
    };

    $scope.handleVisibilityChange = () => {
      if (!document[$scope.hiddenEventName]) $scope.getExistingProcesses();
    };

    $scope.conversionIsDisabled = () => {
      if ($scope.conversionDisabledCause === undefined) {
        $scope.conversionDisabledCause = [];
      } else {
        $scope.conversionDisabledCause.splice(0, $scope.conversionDisabledCause.length);
      }

      if (
        $scope.dbselector === undefined ||
        $scope.dbselector.choice === undefined ||
        $scope.dbselector.choice.length === 0
      ) {
        $scope.conversionDisabledCause.push(
            $filter('translate')('tools.majicedigeo.edigeo.nodatabaseselected')
        );
      }

      if ($scope.radEdigeoSource.choice === 'web') {
        if ($scope.edigeoCommune.list.length === 0) {
          $scope.conversionDisabledCause.push(
              $filter('translate')('tools.majicedigeo.edigeo.nocommuneselected')
          );
        }
      } else {
        if ($scope.edigeoFiles === undefined || $scope.edigeoFiles.length === 0) {
          $scope.conversionDisabledCause.push(
              $filter('translate')('tools.majicedigeo.edigeo.noedigeofile')
          );
        }

        if ($scope.departement === undefined || $scope.departement.choice === undefined) {
          $scope.conversionDisabledCause.push(
              $filter('translate')('tools.majicedigeo.edigeo.nodeptselected')
          );
        }
      }

      if (($scope.edigeoSridNeeded || $scope.departement.choice > 100) &&
        $scope.edigeoSridValue === undefined) {
        $scope.conversionDisabledCause.push(
          $filter('translate')('tools.majicedigeo.edigeo.sridIsNeeded')
        );
      }

      return ($scope.conversionDisabledCause.length !== 0 || $scope.conversionIsRunning);
    };
    $scope.hasCustomEdigeoImportConfig = false;

    $scope.edigeoComponentsConstants = {
      type: 'edigeo',
      appname: 'kis_home',
      name: 'edigeo-import-conf',
    };

    /**
     * Recharge la liste de composants du dernier chargement effectué
     * @param resetSuffix est true si on souhaite ne pas préserver le suffixe utilisé lors du dernier chargement
     */
    $scope.loadCustomComponentList = (resetSuffix = false) => {
      ConfigFactory.get(
        $scope.edigeoComponentsConstants.type,
        $scope.edigeoComponentsConstants.name,
        $scope.edigeoComponentsConstants.appname
      ).then(
        (res) => {
          if (
            angular.isDefined(res) &&
            res.status === 200 &&
            angular.isDefined(res.data) &&
            Array.isArray(res.data) &&
            0 !== res.data.length &&
            res.data.code !== 403
          ) {
            $scope.edigeoComponents.list = resetSuffix ? res.data.map(edigeoComponent => {
              edigeoComponent.destination = edigeoComponent.source;
              return edigeoComponent;
            }) : res.data;
          }
        },
        (reason) => {
          console.error(
            'Configuration des composants edigeo non chargée: ' + reason
          );
        }
      );
    };
    $scope.loadCustomComponentList(true);
    $scope.isTextComponentLoaded = false;

    /**
     * Save component selection
     * Display warning about 'NUMVOIE' & 'PARCELLE_txt'
     */
    $scope.updateEdiCompList = (comp) => {
      const senddata = $scope.edigeoComponents.list;
      if (comp.source === 'NUMVOIE' || comp.source === 'PARCELLE_txt') {
        $scope.isTextComponentLoaded = comp.toLoad;
      }
      // Si on coche PARCELLE alors on coche aussi COMMUNE
      // Si on décoche COMMUNE alors on décoche aussi PARCELLE
      toggleParcelleAndCommune(comp);

      ConfigFactory.add(
        senddata,
        $scope.edigeoComponentsConstants.type,
        $scope.edigeoComponentsConstants.name,
        $scope.edigeoComponentsConstants.appname
      ).then(
        (res) => {
          if (res.status === 200 && res.data) {
            console.log('paramètres de conversion edigeo sauvegardés');
          }
        },
        (reason) => {
          console.error(
            'Impossible de sauvegarder les paramètres de conversion edigeo' +
              reason
          );
        }
      );
    };

    $scope.stopEdigeo = () => {
      $scope.hasInterrupted = true;
      processFactory.getProcessByUsertype(TYPE).then((res) => {
        if (res.data && res.data.length > 0) {
          for (const process of res.data) {
            if (process.etat === 'RUNNING') {
              $scope.conversionIsRunning = $scope.loadingLaunched = true;
              if (process.both) delete process.both;
              ImportExportFactory
              .stopEdigeo(process.uid)
              .then((resStop) => {
                if (resStop && resStop.data){
                  $scope.hasInterrupted = false;
                  if (stop){
                    $interval.cancel(stop);
                    $scope.conversionIsRunning = $scope.loadingLaunched = false
                    $scope.communesBatchImported = null;
                  }
                  require('toastr').warning(
                    $filter('translate')('tools.majicedigeo.edigeo.interrupted')
                  );
                }
              })
              .catch((error) => {
                $scope.hasInterrupted = false;
                require('toastr').error(
                  $filter('translate')('tools.majicedigeo.edigeo.noprocess') + error
                );
              });
            }
          }
        }
      });
    };

      /**
       *     Gestion du  cas où la chargement des EDIGEOs a échoué.
       */
      const loadingEdigeoInterrupted = (res, repeat) => {
        if (
          res.data.errorMessage != null &&
          res.data.errorMessage.indexOf('Use -s_srs to set one') !== -1
        ) {
          $scope.edigeoSridNeeded = true;
          $scope.edigeoSridValue = undefined;
        }

        $interval.cancel(repeat);
        require('toastr').warning($filter('translate')('common.cancel'));
        $scope.setStep(res.data, 'finished');
        $scope.conversionStopped(res.data);
      }

    /**
     * Au changement de la valeur du suffixe à appliquer aux noms de composant,
     * ouvre la popup pour avertir l'utilisateur des risques liés au changement de suffixe.<br>
     */
    $scope.openWarnOrResetSuffix = () => {
      const suffixHasChanged = $scope.existingSuffix !== String($scope.edigeoComponents.suffix);
      const isSuffixEmpty = $scope.edigeoComponents.suffix === undefined || $scope.edigeoComponents.suffix === null || String($scope.edigeoComponents.suffix).length === 0;
      const isYearSelected = $scope.yearFromList !== null && String($scope.yearFromList) !== '';
        if ($scope.componentHasSuffix && isSuffixEmpty) {
          // lorsque le suffixe est effacé en présence d'un suffixe déjà appliqué
          $scope.componentListBuildDestinations();
          $scope.componentHasSuffix = false;
        } else if (($scope.mode === 'text' && !isSuffixEmpty && suffixHasChanged) || ($scope.mode === 'list' && isYearSelected)) {
          ngDialog.open({
            template:
                'js/XG/modules/tools/views/modals/modal.edigeo.suffix.change.warn.html',
            className:
                'ngdialog nopadding noclose edigeo-suffix-confirm ngdialog-overlay ngdialog-theme-plain width700',
            closeByDocument: false,
            scope: $scope,
            closeOnEscape : false
          });
        }
      };

    /**
     * Après changement de la valeur du suffixe à appliquer aux noms de composant,<br>
     * au clic sur "Poursuivre" ou "Annuler" de la popup d'avertissement.<br>
     * Re-initialise le suffixe et lance l'écriture des noms de composants de destination.
     * @param {boolean} hasConfirm est true au clic sur le bouton "Poursuivre"
     */
    $scope.onSuffixConfirm = (hasConfirm) => {
      if (!hasConfirm) {
        if ($scope.mode === 'list') {
          $scope.yearFromList = '';
        } else {
          $scope.edigeoComponents.suffix = '';
          $scope.existingSuffix = '';
        }
      } else {
        $scope.componentHasSuffix = true;
        $scope.existingSuffix = String($scope.edigeoComponents.suffix).slice();
      }
      $scope.componentListBuildDestinations();
    };

    /**
     * Vérifie si le composant <code>componentName</code> est coché ou non par l'utilisateur.
     * @param {string} componentName nom du composant à vérifier la présence dans le tableau de composants de la directive
     * @param {boolean} shouldBeLoaded est true quand on souhaite vérifier le cochage, false si on souhaite vérifier le décochage
     * @return {number} rang du composant dans le tableau de composants de la directive, -1 si absent
     */
    const getComponentIndexByLoad = (componentName, shouldBeLoaded) => {
      const componentIndex = $scope.edigeoComponents.list.findIndex(
          comp => comp.source === componentName);
      if (componentIndex > -1 && $scope.edigeoComponents.list[componentIndex].toLoad === shouldBeLoaded) {
        return componentIndex;
      } else {
        return -1;
      }
    };

    /**
     * Dans la liste des composants EDIGEO à importer : <ul><li>
     * si on coche PARCELLE alors on coche aussi COMMUNE</li><li>
     * si on décoche COMMUNE alors on décoche aussi PARCELLE</li></ul>
     * @param {object} edigeoComponent objet correspondant à un composant edigeo
     */
    const toggleParcelleAndCommune = (edigeoComponent) => {
      if (edigeoComponent.source === 'PARCELLE' && edigeoComponent.toLoad) {
        const communeIndex = getComponentIndexByLoad('COMMUNE', false);
        if (communeIndex > -1) {
          $scope.edigeoComponents.list[communeIndex].toLoad = true;
        }
      }
      if (edigeoComponent.source === 'COMMUNE' && !edigeoComponent.toLoad) {
        const parcelleIndex = getComponentIndexByLoad('PARCELLE', true);
        if (parcelleIndex > -1) {
          $scope.edigeoComponents.list[parcelleIndex].toLoad = false;
        }
      }
    };
  };

  EdigeoCtrl.$inject = [
    '$scope',
    'ngDialog',
    '$filter',
    'FilesFactory',
    'ImportExportFactory',
    'DataStoreFactory',
    'processFactory',
    'ngTableParams',
    '$interval',
    'gaDomUtils',
    'FeatureAttachmentFactory',
    '$timeout',
    'ParametersFactory',
    'SridFactory',
    'gaJsUtils',
    'DecoupAdminFactory',
    'ConfigFactory',
  ];
  return EdigeoCtrl;
});
