'use strict';

/**
 * Factory pour la génération d'un documents décrivant un chantier
 *
 * Cette factory fournit des méthodes pour :
 * - Générer un document DOCX à partir des données d'un chantier
 * - Préparer et formater les données pour la génération du document
 * - Gérer les URLs des cartes WMS et les vues associées
 * - Gérer les composants et styles des cartes
 * - Effectuer des traitements spécifiques sur les fonctionnalités liées
 *
 * @module XG/widgets/utilities/form/services/IndigauPrintChantierFactory
 * @requires module:FeatureTypeFactory
 * @requires module:gaDomUtils
 * @requires module:QueryFactory
 * @requires module:AlertHpoFactory
 * @requires module:SirocoFactory
 * @requires module:DataStoreFactory
 * @requires $q
 * @requires module:IndigauEnvFactory
 * @requires $location
 * @requires module:ActionsManager
 * @requires module:DocumentFactory
 */

define(function() {
  // Constantes pour les tailles d'images
  const IMAGE_SIZES = {
    MAIN: '880,726',
    PREVIEW1: '600,600',
    PREVIEW2: '180,180',
    SCALE_FACTOR_MAIN: 800,
    SCALE_FACTOR_PREVIEW: 600
  };

  // Noms des couches par défaut
  const LAYER_NAMES = {
    DEFAULT: 'siroco_chantier%2Caep_cana%2CAEP_ANOMALIE',
    STYLES: 'chantier1%2CAEP_CANA_IMPORTANCE%2Canomalie',
    STYLES_ALT: 'chantier1%2CAEP_CANA_IMPORTANCE%2Anomalie'
  };

  // Paramètres d'URL par défaut
  const URL_PARAMS = {
    CQL_FILTER: 'CQL_FILTER=1%3D1',
    CQL_FILTER_REPLACEMENT: 'CQL_FILTER=fid%3D',
    TYPE_PARAM: '&type',
    TYPE_REPLACEMENT: ';1%3D1;1%3D1;1%3D1;1%3D1&type'
  };

  const IndigauPrintChantierFactory = function(FeatureTypeFactory, gaDomUtils, QueryFactory,
    AlertHpoFactory, SirocoFactory, DataStoreFactory, $q, IndigauEnvFactory, $location,
    ActionsManager, DocumentFactory) {

    /**
     * Récupère l'URL WMS pour un chantier donné
     * @param {Object} chantier - L'objet représentant le chantier
     * @param {boolean} crsOk - Indique si le système de coordonnées est correct
     * @returns {Promise} Une promesse résolue avec l'URL WMS
     */
    const getWmsMapUrl = function(chantier, crsOk) {
      const deferred = $q.defer();

      //-- Le chantier est-il dans le bon sustème de projection ?
      const descCana = FeatureTypeFactory.getFeatureByUid(
        chantier.properties.FTIUID
      );

      // Récupération de l'étendue dans le système de projection Web Mercator (EPSG:3857)
      FeatureTypeFactory.getExtent(descCana.uid, 'EPSG:3857').then((res) => {
        const canaBb = res.data;
        if (crsOk === false) {
          // Si le système de coordonnées n'est pas correct,
          // on récupère la feature depuis la base de données
          QueryFactory.get(
            IndigauEnvFactory.INDIGAU_FTIS['chantiers'].uid,
            chantier.id,
            'EPSG:3857'
          ).then((res) => {
            const chantierFromDb = res.data.features[0];
            // Construction de l'URL WMS pour la carte avec les données corrigées
            deferred.resolve(
              SirocoFactory.getWmsMapUrl1(chantierFromDb, canaBb)
            );
          });
        }
        //-- L'appelant dit que oui: on peut préparer l'url de la carte de suite.
        else {
          // Si le système de coordonnées est correct,
          // on peut préparer l'url de la carte de suite.
          deferred.resolve(SirocoFactory.getWmsMapUrl1(chantier, canaBb));
        }
      }, (res) => {
        deferred.reject(res);
      });

      return deferred.promise;
    };


    /**
     * Crée une URL Esri pour une requête d'obtention de dessin d'une carte
     * @param {string[]} components - Liste des composants à inclure dans l'URL
     * @param {string} createEsriUrl - URL de base pour les services Esri
     * @param {string} storeName - Nom du store Esri
     * @returns {Promise<string>} Promise résolue avec l'URL Esri
     */
    const createEsriUrl = (components,createEsriUrl,storeName) => {
      const defer = $q.defer();
      if(components && createEsriUrl && storeName && components.length != 0){
        DataStoreFactory.getEsriUrl(storeName).then((res) => {
          let urlEsri = res.data + '/export?bboxSR=3857&layerDefs=&imageSR=&historicMoment='
            + '&format=png&transparent=true&dpi=&time=&layerTimeOptions=&dynamicLayers='
            + '&gdbVersion=&mapScale=&rotation=&datumTransformations=&layerParameterValues='
            + '&mapRangeValues=&layerRangeValues=&f=image';
          urlEsri += '&layers=show:' + components.join(',');
          DataStoreFactory.getEsriToken(storeName).then((res1) => {
            urlEsri += '&token='+res1.data;
            defer.resolve(urlEsri);
          },() => {
            defer.resolve(urlEsri);
          });
        },() => {
          defer.resolve('');
        });
      }
      else {
        defer.resolve('');
      }
      return defer.promise;
    };


    /**
     * calculer_chantier from the chantier form (intervention)
     */
    const specialTreatement = (relationTreatement) => {
      if (
        relationTreatement &&
                relationTreatement.features &&
                relationTreatement.features.length > 0
      ) {
        const relationsChantiers = {};
        const relations = [];
        relationTreatement.features.map((x) => {
          if (x && x.id && IndigauEnvFactory.INDIGAU_FTIS['chantiers_relations']
            && x.id.indexOf(IndigauEnvFactory.INDIGAU_FTIS['chantiers_relations'].name) === -1) {
            relations.push(x);
          } else if (x && x.id && IndigauEnvFactory.INDIGAU_FTIS['chantiers_relations']
            && x.id.indexOf(IndigauEnvFactory.INDIGAU_FTIS['chantiers_relations'].name) !== -1
            && x.properties && x.properties.FEATID) {
            relationsChantiers[x.properties.FEATID] = {
              HPO_CALC_CASSE: x.properties.CASSE,
              HPO_CALC_REPARATION: x.properties.COUT_REPARATION,
              HPO_CALC_RENOUVELLEMENT: x.properties.COUT_RENOUVELLEMENT,
              HPO_CALC_RENDEMENT: x.properties.RENDEMENT,
              HPO_CALC_NOTE_PRIORITE: x.properties.NOTE_PRIORITE,
            };
          }
        });
        relations.map((x) => {
          if (x && x.id && Object.prototype.hasOwnProperty.call(relationsChantiers, x.id))
            Object.assign(x.properties, relationsChantiers[x.id]);
        });
        return {
          type: 'FeatureCollection',
          crs: relationTreatement.crs,
          totalFeatures: relations.length,
          features: relations,
        };
      } else {
        return relationTreatement;
      }
    };


    /**
     * Crée des fonctionnalités spéciales liées au chantier
     * @param {Object} relatedfeature - Objet contenant les fonctionnalités liées
     * @param {Object} sendata - Objet contenant les données à formater
     * @param {Object} currentResourceConfig - Configuration du type de ressource
     * @returns {void}
     */
    const createSpecialRelatedFeature = (relatedfeature,sendata,currentResourceConfig) => {
      for (let i = 0;i< relatedfeature.features.length;i++) {
        const diametre = relatedfeature.features[i].properties[currentResourceConfig.diametre];
        let feature = undefined;
        if(sendata.SPECIAL_RELATED_FEATURES.length>0){
          for(let j = 0;j< sendata.SPECIAL_RELATED_FEATURES.length;j++){
            if(sendata.SPECIAL_RELATED_FEATURES[j][currentResourceConfig.diametre] == diametre){
              feature = sendata.SPECIAL_RELATED_FEATURES[j];
              break;
            }
          }
        }
        if(feature == undefined){
          relatedfeature.features[i].properties.chantierUid = currentResourceConfig.uid;
          sendata.SPECIAL_RELATED_FEATURES.push(relatedfeature.features[i].properties);
        }else{
          const featureCurrentResourceConfig
            = IndigauEnvFactory.configHpo().currentResources
              .filter((chantier) => chantier.uid==feature.chantierUid)[0];
          feature['HPO_CALC_NOTE_PRIORITE']
            = (feature['HPO_CALC_NOTE_PRIORITE'] * feature[featureCurrentResourceConfig.longueur]
            + relatedfeature.features[i].properties['HPO_CALC_NOTE_PRIORITE']
            * relatedfeature.features[i].properties[currentResourceConfig.longueur])
            / (feature[featureCurrentResourceConfig.longueur]
            + relatedfeature.features[i].properties[currentResourceConfig.longueur]);
          feature[featureCurrentResourceConfig.longueur]
            = feature[featureCurrentResourceConfig.longueur]
            + relatedfeature.features[i].properties[currentResourceConfig.longueur];
          feature['HPO_CALC_RENOUVELLEMENT']
            = feature['HPO_CALC_RENOUVELLEMENT']
            + relatedfeature.features[i].properties['HPO_CALC_RENOUVELLEMENT'];
        }
      }
    };


    /**
     * Calcule et formate les données d'envoi pour la génération du document
     * @param {Object} sendata - Objet contenant les données à formater
     * @param {Object} sendata.current - Données courantes du chantier
     * @param {Object} sendata.current.properties - Propriétés du chantier
     * @param {number} sendata.current.properties.COUT_RENOUVELLEMENT_EXACT -
     *   Coût exact de renouvellement
     * @param {number} sendata.current.properties.DIVERS - Pourcentage de frais divers
     * @param {number} sendata.current.properties.DIVERS_VALUE - Valeur calculée des frais divers
     * @param {number} sendata.current.properties.COUT_RENOUVELLEMENT - Coût total de renouvellement
     * @param {Array} sendata.SPECIAL_RELATED_FEATURES - Tableau des fonctionnalités liées
     * @param {string} uid - Identifiant unique du type de ressource
     * @returns {void}
     */
    const calculateAndFormatSendData = (sendata, uid) => {
      const currentResourceConfig
        = IndigauEnvFactory.configHpo().currentResources
          .filter((chantier) => chantier.uid==uid)[0];
      sendata.SPECIAL_RELATED_FEATURES.sort(
        (a, b) => parseFloat(a[currentResourceConfig.diametre])
          - parseFloat(b[currentResourceConfig.diametre]));
      //Arrondir les valeurs au milier pret et calculer la somme
      sendata.current.properties.COUT_RENOUVELLEMENT_EXACT = 0 ;
      for (let i = 0;i< sendata.SPECIAL_RELATED_FEATURES.length;i++) {
        sendata.SPECIAL_RELATED_FEATURES[i].HPO_CALC_RENOUVELLEMENT
          = Math.round(sendata.SPECIAL_RELATED_FEATURES[i].HPO_CALC_RENOUVELLEMENT/1000)*1000;
        sendata.current.properties.COUT_RENOUVELLEMENT_EXACT
          += sendata.SPECIAL_RELATED_FEATURES[i].HPO_CALC_RENOUVELLEMENT;
      }
      if(sendata.current.properties.DIVERS == null
        || !angular.isDefined(sendata.current.properties.DIVERS)){
        sendata.current.properties.DIVERS = 0;
      }
      sendata.current.properties.DIVERS_VALUE =
        sendata.current.properties.COUT_RENOUVELLEMENT_EXACT
        * sendata.current.properties.DIVERS/ 100;
      sendata.current.properties.COUT_RENOUVELLEMENT
        = sendata.current.properties.COUT_RENOUVELLEMENT_EXACT
          + sendata.current.properties.DIVERS_VALUE;
      sendata.hpoConfig = IndigauEnvFactory.configHpo();
    };


    /**
     * Récupère et organise les composants géographiques pour la génération du document
     * @param {string[]} components - Tableau des noms des composants standards
     * @param {string[]} arcGisComponents - Tableau des identifiants des composants ArcGIS
     * @param {string[]} componentsStyles - Tableau des styles associés aux composants
     * @param {Object} ftiCana - Objet représentant le type de composant principal
     * @param {string} ftiCana.type - Type du composant ('esri' ou autre)
     * @param {string} ftiCana.ogcId - Identifiant OGC pour les services Esri
     * @param {string} ftiCana.wms - URL du service WMS pour les composants Esri
     * @param {string} ftiCana.storeName - Nom du store pour les composants Esri
     * @param {string} ftiCana.name - Nom du composant
     * @param {string} ftiCana.defaultStyle - Style par défaut du composant
     * @param {Array} ftiCana.relations - Liste des relations du composant
     * @param {Object} agInfo - Objet pour stocker les informations spécifiques à ArcGIS
     * @param {string} agInfo.esriUrl - URL du service Esri
     * @param {string} agInfo.esriStoreName - Nom du store Esri
     * @returns {void}
     */
    const getComponents = (components, arcGisComponents, componentsStyles, ftiCana, agInfo) => {

      if(ftiCana.type == 'esri'){
        arcGisComponents.push(ftiCana.ogcId);
        agInfo.esriUrl = ftiCana.wms;
        agInfo.esriStoreName = ftiCana.storeName;
      }
      else{
        components.push(ftiCana.name);
        componentsStyles.push(ftiCana.defaultStyle || '');
      }
      ftiCana.relations.map((x) => {
        if (x.type === 'REL_SIMPLE' && x.idEnd) {
          const ftiend = FeatureTypeFactory.getFeatureByUid(
            x.idEnd
          );
          if (ftiend.geographic) {
            if(ftiCana.type == 'esri'){
              arcGisComponents.push(ftiend.ogcId);
            }
            else{
              components.push(ftiend.name);
              componentsStyles.push(ftiend.defaultStyle || '');
            }
          }
        }
      });
    };


    /**
     * Construit et met à jour les URLs nécessaires pour la génération du document
     * @param {string[]} components - Liste des composants à inclure dans l'URL
     * @param {string[]} componentsStyles - Liste des styles des composants
     * @param {string} esriurl - URL de base pour les services Esri
     * @param {Object} chantierFeature - Objet représentant le chantier
     * @param {string} chantierFeature.id - Identifiant unique du chantier
     * @param {Object} url - Objet contenant les différentes URLs à mettre à jour
     * @param {string} url.plan - URL du plan principal
     * @param {string} url.plan_prev01 - URL de la première prévisualisation
     * @param {string} url.plan_prev02 - URL de la deuxième prévisualisation
     * @returns {void}
     */
    const urlsForDoc = (components, componentsStyles, esriurl, chantierFeature, url) => {
      // Mise à jour de l'URL principale
      url.plan = url.plan
        .replace(LAYER_NAMES.DEFAULT, components.join(','))
        .replace(LAYER_NAMES.STYLES, componentsStyles.join(','))
        .replace(
          URL_PARAMS.CQL_FILTER,
          `${URL_PARAMS.CQL_FILTER_REPLACEMENT}${chantierFeature.id.split('.')[1]}`
        );

      // Gestion du cas où il y a plus de 3 composants
      if (components.length > 3) {
        url.plan = url.plan.replace(
          URL_PARAMS.TYPE_PARAM,
          URL_PARAMS.TYPE_REPLACEMENT
        );
      }

      // Construction des URLs pour les différentes vues
      const bboxParam = url.plan.substring(url.plan.indexOf('BBOX='));
      url.planEsri = esriurl ?
        `${esriurl}&size=${IMAGE_SIZES.MAIN}&${bboxParam}` : '';

      // Mise à jour des URLs de prévisualisation
      const updatePreviewUrls = (source, target, size) => {
        target = source
          .replace(LAYER_NAMES.DEFAULT, `${components[0]},${components[1]}`)
          .replace(
            LAYER_NAMES.STYLES_ALT,
            `${componentsStyles[0]},${componentsStyles[1]}`
          );

        const previewBbox = target.substring(target.indexOf('BBOX='));
        const previewEsri = esriurl ?
          `${esriurl}&size=${size}&${previewBbox}` : '';

        return { preview: target, previewEsri };
      };

      // Mise à jour des URLs de prévisualisation 1 et 2
      const prev01 = updatePreviewUrls(
        url.plan_prev01,
        'plan_prev01',
        IMAGE_SIZES.PREVIEW1
      );
      const prev02 = updatePreviewUrls(
        url.plan_prev02,
        'plan_prev02',
        IMAGE_SIZES.PREVIEW2
      );

      Object.assign(url, {
        plan_prev01: prev01.preview,
        plan_prev01Esri: prev01.previewEsri,
        plan_prev02: prev02.preview,
        plan_prev02Esri: prev02.previewEsri
      });
      url.locator02 =
        $location.absUrl().replace($location.url(), '/').replace('#/', '') + 'img/hpo/locator2.png';
      url.locator01 =
        $location.absUrl().replace($location.url(), '/').replace('#/', '') + 'img/hpo/locator1.png';
    };


    /**
     * Prépare les données à envoyer pour la génération du document
     * @param {Object} chantierFeature - L'objet représentant le chantier
     * @param {Object} result - Le résultat contenant les données du chantier
     * @param {Object} relatedfeature - Les fonctionnalités liées au chantier
     * @param {Object} url - Les URLs des différentes vues du plan
     * @param {string} url.fdplan - URL du plan principal contenant les coordonnées BBOX
     * @param {string} url.fdplan_prev01 - URL de la prévisualisation du plan
     *  contenant les coordonnées BBOX
     * @returns {Object} Les données formatées pour la génération du document
     */
    const prepareDataToSend = (chantierFeature, result, relatedfeature, url) => {
      const sendataOriginal = ActionsManager.getPortalDoc(
        chantierFeature,
        result,
        IndigauEnvFactory.INDIGAU_FTIS['chantiers'],
        relatedfeature
      );
      const sendata = angular.copy(sendataOriginal);
      const coordinateplan = url.fdplan.substring(url.fdplan.indexOf('BBOX=')+5)
        .split('%2C');
      const coordinateplan_prev01 = url.fdplan_prev01
        .substring(url.fdplan_prev01.indexOf('BBOX=')+5).split('%2C');

      const xDiffPlan = Math.abs(parseFloat(coordinateplan[2]) - parseFloat(coordinateplan[0]));
      const xDiffPreview = Math.abs(parseFloat(coordinateplan_prev01[2]) -
        parseFloat(coordinateplan_prev01[0]));

      sendata.current.properties.SCALE_LINE_PLAN =
        IMAGE_SIZES.SCALE_FACTOR_MAIN / xDiffPlan;
      sendata.current.properties.SCALE_LINE_PLAN_PREV01 =
        IMAGE_SIZES.SCALE_FACTOR_PREVIEW / xDiffPreview;
      sendata.SCALE_LINE = IMAGE_SIZES.SCALE_FACTOR_MAIN / xDiffPlan;
      sendata.SCALE_LINE01 = IMAGE_SIZES.SCALE_FACTOR_PREVIEW / xDiffPreview;

      sendata.SPECIAL_RELATED_FEATURES=[];
      return sendata;
    };


    /**
     * Génère un document DOCX et lance son téléchargement
     * @param {Object} dataToSend - Les données à inclure dans le document
     * @param {Object} chantierFeature - L'objet contenant les informations du chantier
     * @param {string} chantierFeature.id - L'identifiant unique du chantier
     * @returns {void}
     */
    const generateDocx = (dataToSend, chantierFeature) => {
      DocumentFactory.generatedocx(
        dataToSend,
        IndigauEnvFactory.rapportName(),
        chantierFeature.id
      ).then(
        (res) => {
          const portalId = angular.module('gcMain').portalid;
          window.open(`/services/${portalId}/documents/downloaddocx?name=${res.data.name}`);
          gaDomUtils.hideGlobalLoader();
        },
        (res) => {
          gaDomUtils.hideGlobalLoader();
          AlertHpoFactory.showErrorMessage(res);
        }
      );
    };


    /**
     * Imprime un chantier
     * @param {Object} dataTablRes - Les données du chantier
     * @returns {void}
     */
    const printChantier = (dataTablRes) => {
      gaDomUtils.showGlobalLoader();
      let chantierFeature;
      for (const prop in dataTablRes) {
        chantierFeature = dataTablRes[prop][0];
        break;
      }
      const chantierUid = IndigauEnvFactory.INDIGAU_FTIS['chantiers'].uid;
      QueryFactory.relation(chantierUid, '*', chantierFeature.id, '').then(
        (res1) => {
          let relatedfeature = res1.data;
          getWmsMapUrl(chantierFeature, true).then(
            (url) => {
              try {
                const result = {
                  type: 'FeatureCollection',
                  features: [chantierFeature],
                };
                const arcGisComponents = [];
                const agInfo = {};
                const components = [IndigauEnvFactory.INDIGAU_FTIS['chantiers'].name];
                const componentsStyles = [
                  IndigauEnvFactory.INDIGAU_FTIS['chantiers'].defaultStyle || '',
                ];
                const ftiCana = FeatureTypeFactory.getFeatureByUid(
                  chantierFeature.properties.FTIUID
                );
                getComponents(components,componentsStyles, arcGisComponents, ftiCana, agInfo);
                createEsriUrl(arcGisComponents,agInfo.esriUrl,agInfo.esriStoreName)
                  .then((esriurl) => {
                    urlsForDoc(components, componentsStyles, esriurl, chantierFeature, url);
                    relatedfeature = specialTreatement(relatedfeature);
                    const dataToSend
                      = prepareDataToSend (chantierFeature, result, relatedfeature, url);
                    const currentResourceConfig
                      = IndigauEnvFactory.configHpo().currentResources.filter(
                        (chantier) => chantier.uid==chantierFeature.properties.FTIUID)[0];
                    createSpecialRelatedFeature(relatedfeature,dataToSend,currentResourceConfig);
                    calculateAndFormatSendData(dataToSend,chantierFeature.properties.FTIUID);
                    Object.assign(dataToSend, url);
                    dataToSend.hpoFtiuid = ftiCana.uid;
                    for (const relation of IndigauEnvFactory.INDIGAU_FTIS['chantiers'].relations) {
                      if (relation.type === 'REL_NM' && relation.idEnd === ftiCana.uid) {
                        dataToSend.REL_NAME = relation.name;
                        break;
                      }
                    }
                    generateDocx(dataToSend, chantierFeature);
                  });
              }
              catch (e) {
                e.stack;
                gaDomUtils.hideGlobalLoader();
              }
            },
            (res) => {
              gaDomUtils.hideGlobalLoader();
              AlertHpoFactory.showErrorMessage(res);
            }
          );
        },
        (res) => {
          gaDomUtils.hideGlobalLoader();
          AlertHpoFactory.showErrorMessage(res);
        }
      );
    };


    return {
      printChantier: printChantier
    };
  };

  IndigauPrintChantierFactory.$inject = ['FeatureTypeFactory', 'gaDomUtils', 'QueryFactory',
    'AlertHpoFactory', 'SirocoFactory', 'DataStoreFactory', '$q', 'IndigauEnvFactory', '$location',
    'ActionsManager', 'DocumentFactory'
  ];
  return IndigauPrintChantierFactory;
});