'use strict';
define(function () {
  var homepageCarousel = function (
      $window,
      $timeout,
      $rootScope,
      ConfigFactory,
      ApplicationFactory,
      FilesFactory,
      $filter,
      $location,
      gaJsUtils,
      homeFactory,
      ngDialog,
      gaDomUtils,
      PortalsFactory
  ) {
    return {
      templateUrl: 'js/XG/modules/home/views/directives/homepageCarousel.html',
      restrict: 'EA',
      scope: {
        items: '=',         // (OBLIGATOIRE) tableau des items du carousel (groupes ou liens)
        mode: '=',          // (OBLIGATOIRE) mode = 'config' au clic sur le bouton de configuration
        halign: '=?',       // alignement horizontal du carousel. 'left' si undefined
        isPrimary: '=?',    // true pour le carousel des groupes, false/undefined pour celui des liens
        primaryId: '=?',    // entier identifiant le groupe (groups[i].id). Obligatoire pour les liens
        images: '=?',       // liste d'images disponibles pour mettre dans un groupe/lien
        portalId: '=?',     // uid identifiant du portail pour goToApp. Obligatoire pour les liens
        adminItem: '=?',    // item renvoyant vers la page d'admin
        onOrderChange: '=?'// callback appeller lors du drag&drop, pour pouvoir sauvegarder l'ordre des items
      },
      link: function (scope, element) {

        // objet de la dropzone
        scope.dzMemShare = {}

        /**
         * activation du drag and drop
         */
        scope.$watch('items', () => {
          if (scope.mode === 'config') {
            // active drag/drop
            // On doit attendre qui l'interface se mette à jour pour avoir une liste de theme à jour
            $timeout(addDragDropEvents);
          }
        });

        /**
         * Au clic sur une icône du gras pour le titre et la description du titre,
         * change la propriété weight de la police et
         * modifie la propriété temporaire "isBold" pour répercuter l'état sur le bouton
         * @param {string} inputText 'title' ou 'desc'
         */
        scope.onBoldChange = (inputText) => {
          if (scope.edit_resource) {
            homeFactory.toggleBoldByWeight(scope.edit_resource[inputText]);
          }
        };

        /**
         * Au clic sur le bouton "Changer l'alignement vertical du texte".
         * Change l'alignement vertical du titre du groupe/thème
         */
        scope.onValignChange = () => {
          if (scope.edit_resource && homeFactory.vAlignPositions && Array.isArray(
              homeFactory.vAlignPositions) && homeFactory.vAlignPositions.length > 0) {
            const oldPosition = scope.edit_resource.title.valign;
            let nextPosition = 1;
            if (homeFactory.vAlignPositions.includes(oldPosition)) {
              nextPosition = homeFactory.vAlignPositions.findIndex(p => p === oldPosition) + 1;
              if (nextPosition > homeFactory.vAlignPositions.length - 1) {
                nextPosition = 0;
              }
            } else {
              nextPosition = 0;
            }
            scope.edit_resource.title.valign = homeFactory.vAlignPositions[nextPosition];
          }
        };

        /**
         * Au clic sur le bouton "Alignement horizontal" du titre ou de la description
         * d'un groupe/lien dans la popup d'ajout/éditon
         * @param {string} inputType 'title' ou 'desc'
         */
        scope.onHalignChange = (inputType) => {
          if (homeFactory.hasOwnProperty('hAlignPositions') && Array.isArray(
                  homeFactory.hAlignPositions)
              && homeFactory.hAlignPositions.length > 0) {
            // toggle 'left' -> 'center' -> 'right' -> 'left' -> ...
            homeFactory.setHalign(scope.edit_resource[inputType]);
          } else {
            console.error($filter('translate')('portals.homepage.halignError'))
          }
        };

        /**
         * Au clic sur le bouton toggle "Forme de l'item" (cercle/carré)
         * dans la popup d'ajout/édition d'un groupe/lien
         */
        scope.onShapeChange = () => {
          scope.edit_resource.shape = scope.edit_resource.isCircle ? 'circle' : 'rectangle';
        };

        /**
         * Au clic sur une image dans la mosaïque d'images de la popup d'ajout/édition d'un groupe/lien,
         * sélectionne l'image, et met à jour la propriété 'image' du groupe/lien édité
         * @param {number} indexOfNewImage rang du groupe/lien dans le tableau de groupes/liens de la directive
         */
        scope.changeImage = (indexOfNewImage) => {
          if (Number.isInteger(indexOfNewImage) && indexOfNewImage > -1
              && Array.isArray(scope.images) && indexOfNewImage < scope.images.length) {
            const previousImageIndex = scope.images.findIndex(image => image.active);
            // si on a cliqué sur une image qui n'était pas selectionné
            if (previousImageIndex !== indexOfNewImage) {
              // désactive toutes les images sauf la nouvelle image sélectionné
              scope.images.map( (image, localIndexOfImage) => {
                  image.active = (localIndexOfImage === indexOfNewImage);
                  return image;
                });
              scope.edit_resource.image = scope.images[indexOfNewImage].src;
            } else {
              //désactive toutes les images
              scope.images.map( image => {
                  image.active = false;
                  return image;
                });
              scope.edit_resource.image = null;
            }
          } else {
            console.error($filter('translate')('portals.homepage.onImageChangeError'))
          }
        };

        /**
         * Sélectionne / désélectionne un groupe ou un thème/lien
         * @param {number} index rang de l'item dans le carousel
         */
        scope.selectItem = (index) => {
          if (scope.items && Number.isInteger(index)) {
            if (!scope.items[index].active && scope.isPrimary && !scope.items[index].isadmin) {
              // Sélectionne le nouveau groupe actif et désélectionne les autres
              scope.items.map(it => {
                  it.active = (it.id === scope.items[index].id);
                  return it;
                });
              // sélection du groupe au rang index
              // demande au parent d'actualiser la variable primaryId en entrée du carousel des liens
              scope.$emit('onHomepageGroupSelected', scope.items[index].id);
            }

            // pas d'ouverture de lien en mode configuration pour éviter l'ouverture lors du drag&drop
            if (scope.mode!=='config') {
              // si on est dans le carousel des groupes,
              // on demande au parent d'actualiser le filtrage des liens dans le carousel des liens
              if (scope.isPrimary) {
                if (scope.items[index].isadmin) {
                  scope.goToApp('admin');
                }
              } else {
                // clic sur un item du carousel des liens
                if (scope.items[index].appuid && scope.items[index].appuid.length > 0) {
                  // lien vers une application
                  scope.goToApp(scope.items[index].appuid);
                } else if (scope.items[index].url && scope.items[index].url.length > 0) {
                  // lien externe
                  gaJsUtils.redirectTo(scope.items[index].url);
                }
              }
            }
          }
        };

        /**
         * Supprime un élément existant du carousel au rang donné
         * Enlève 1 à la propriété index des groupes situés à partir du rang donné
         * @param {number} index rang de l'item dans le tableau d'items
         */
        scope.deleteItem = (index) => {
          if (index === 'admin') {
            scope.adminItem = null;
          } else {
            scope.$emit('deleteHomepageItem', {edit_resource: scope.items[index], isGroup: scope.isPrimary});
            for (let i = index; i < scope.items.length; i++) {
              scope.items[i].index = i - 1;
            }
          }
        };

        let editPopup;
        /**
         * Ouvre la popup d'ajout/édition d'un élément :<br>
         * Au clic sur l'item vierge de la liste d'élément (bouton + )
         * créé la variable edit_resource.<br>
         * Au clic sur le bouton édition (crayon) d'un item existant
         * récupère l'item sélectionné suivant son index<br>
         * @param {number} index rang de l'item dans le carousel
         */
        scope.openItem = (index) => {

          // créé la variable scope.edit_resource suivant 'index':
          if (Number.isInteger(index) && index > -1) {
            editItem(index);
          } else if (index === 'admin') {
            editAdminItem();
          } else {
            createItem();
          }

          if (scope.isPrimary) {
            // si c'est un groupe on lui affecte une branche à extrémité écrasée
            scope.edit_resource.isCircle = scope.edit_resource.shape === 'circle';
          } else {
            // on définit la propriété temporaire isExternal suivant la présence de la propriété url
            // vérifie seulement l'existence de la propriété url du lien
            scope.edit_resource.isExternal = gaJsUtils.notNullAndDefined(scope.edit_resource.url)
                && scope.edit_resource.url.length > 0;
          }

          // créé la propriété temporaire "isBold" qui est le model du bouton "Police en gras" du titre du groupe/lien
          // actuellement on gére bold/unbold mais en stockant ainsi un nombre dans 'weight'
          // cela laisse la possibilité de gérer plusieurs valeurs par la suite
          scope.edit_resource.isBold = gaJsUtils.notNullAndDefined(scope.edit_resource,
                  'title.weight')
              && scope.edit_resource.title.weight > 400;

          const className = 'ngdialog-theme-plain width900 height1000 nopadding kis-home-news-carousel-dialog miniclose';
          editPopup = ngDialog.open({
            template: 'js/XG/modules/home/views/modals/modal.home.carousel.html',
            className: scope.isPrimary ? className.replace('height1000', 'height800') : className,
            closeByDocument: false,
            closeByEscape: true,
            scope: scope,
            preCloseCallback: () => {
              scope.edit_resource = null;
            },
          });
        };

        /**
         * Au clic sur le bouton d'édition d'un groupe/lien,
         * créé la variable edit_resource à partir de l'objet du tableau item au rang index et
         * active l'image du groupe/lien dans la mosaïque
         * @param {number} index rang de l'item à éditer dans le tableau d'items de la directive
         */
        const editItem = (index) => {
          // groupe/lien existant
          scope.edit_resource = angular.copy(scope.items[index]);

          // titre variable
          scope.popupTitle = $filter('translate')(
              'portals.homepage.v2.popup.edit' + scope.getItemType(true) + 'Title');

          // active l'image de l'item dans la popup
          initActiveItemImage();
        };

        /**
         * Au clic sur le bouton de nouveau groupe/lien (+) : <ul><li>
         * créé la variable edit_resource à partir d'un template stocké dans le service</li><li>
         * modifie le titre de la popup</li><li>
         * désactive toutes les images</li></ul>
         */
        const createItem = () => {
          // index temporaire dans l'affichage filtré courant
          const nextIndex = getNextIndex();

          // groupe/lien vierge
          scope.edit_resource = scope.isPrimary ? homeFactory.getEmptyGroup(nextIndex)
              : homeFactory.getEmptyLink(nextIndex, scope.groupid);

          // titre variable
          scope.popupTitle = $filter('translate')(
              'portals.homepage.v2.popup.add' + scope.getItemType(true) + 'Title');

          // assure de n'avoir aucune image active dans la popup
          for (const image of scope.images) {
            image.active = false;
          }
        };

        /**
         * Au clic sur le bouton d'édition du groupe "Admin" :<ul><li>
         * créé la variable edit_resource à partir de l'objet adminItem de la page d'accueil</li><li>
         * modifie le titre de la popup</li><li>
         * active l'image de l'item "admin" dans la mosaïque</li></ul>
         */
        const editAdminItem = () => {
          // groupe/lien existant
          scope.edit_resource = angular.copy(scope.adminItem);

          // titre variable
          scope.popupTitle = $filter('translate')(
              'portals.homepage.v2.popup.edit' + scope.getItemType(true) + 'Title');

          // active l'image de l'item dans la popup
          initActiveItemImage();
        };

        /**
         * Au clic sur le bouton valider de l'ajout/édition d'un groupe/lien de page d'accueil,
         * ajoute l'objet dans le tableau de groupes/liens
         */
        scope.saveItem = () => {
          if (scope.edit_resource) {
            gaDomUtils.showGlobalLoader();
            // suppression des propriétés temporaires
            if (scope.edit_resource.hasOwnProperty('isBold')
                && gaJsUtils.notNullAndDefined(scope.edit_resource.isBold)) {
              delete scope.edit_resource.isBold;
            }
            if (scope.edit_resource.hasOwnProperty('isExternal')
                && gaJsUtils.notNullAndDefined(scope.edit_resource.isExternal)) {
              delete scope.edit_resource.isExternal;
            }
            if (scope.edit_resource.hasOwnProperty('isCirculal')
                && gaJsUtils.notNullAndDefined(scope.edit_resource.isCirculal)) {
              delete scope.edit_resource.isCirculal;
            }
            if (!scope.edit_resource.isadmin) {
              if (!Number.isInteger(scope.edit_resource.id)) {
                $timeout(() => {
                  addDragDropEventsLast();
                }, 300);
              }
              scope.$emit('addItemIntoHomepage', {edit_resource: scope.edit_resource, isGroup: scope.isPrimary});
            } else {
              scope.adminItem = angular.copy(scope.edit_resource);
            }
            gaDomUtils.hideGlobalLoader();
          }
          if (editPopup) {
            hideItem();
            editPopup.close();
            for (const image of scope.images) {
              image.active = false;
            }
          }
        };

        /**
         * Masque la preview dans la popup de l'ajout/édition d'un groupe/lien de page d'accueil
         * Utilisé à la fermeture de la popup pour ne pas afficher un item vide.
         */
        const hideItem = () => {
          const previewSide = document.getElementById('preview-side');
          if (previewSide) {
            const item = previewSide.querySelector('.item');
            if (item) {
              item.style.visibility = 'hidden';
            }
          }
        };

        /**
         * Au clic sur le bouton "Annuler" de la popup d'ajout/édition d'un groupe/lien,
         * supprime la variable edit_resource et
         * ferme la popup
         */
        scope.cancelItem = () => {
          if (editPopup) {
            hideItem();
            editPopup.close();
            for (const image of scope.images) {
              image.active = false;
            }
          }
        };

        /**
         * Récupère la liste des applications
         * auxquelles un utilisateur root a le droit
         * d'accéder.
         */
        scope.getUserApplications = () => {
          homeFactory.getApplications().then(
              apps => {
                scope.apps = apps;
              }
          )
        };

        /**
         * Détermine l'alignement vertical du titre de l'élément en fonction du nom de l'image.
         * Traite uniquement les élements dont le titre n'a pas d'alignement vertical initial.
         * Aligne au centre les titres d'élément qui n'auraient pas les propriétés valign et image
         */
        const setValignFromImagesIfEmpty = () => {
          // Quand un texte d'item a un valign null,
          // on vérifie si le nom de l'image svg contient 'top', 'center' ou 'bottom'
          if (scope.items && Array.isArray(scope.items) && scope.items.length > 0) {
            for (const item of scope.items) {

              if (!item.title.valign || item.title.valign.length === 0) {

                // 'center' par défaut
                item.title.valign = homeFactory.vAlignPositions[1];

                if (item.image && item.image.length > 0) {
                  if (item.image.contains('-top')) {
                    // si l'image contient top et l'élement est un groupe => valign = top
                    // si l'image contient top et l'élement est un lien => valign = bottom
                    item.title.valign = scope.isPrimary ? homeFactory.vAlignPositions[0]
                        : homeFactory.vAlignPositions[2];
                  } else if (item.image.contains('-bottom')) {
                    // si l'image contient bottom et l'élement est un groupe => valign = bottom
                    // si l'image contient bottom et l'élement est un lien => valign = top
                    item.title.valign = scope.isPrimary ? homeFactory.vAlignPositions[2]
                        : homeFactory.vAlignPositions[0];
                  }
                }
              }
            }
          }
        };

        /**
         * A l'initialisation de la popup modal_app_carousel,
         * désactive toutes les images et
         * active l'image du groupe/lien édité dans la mosaïque d'images de la bibliothèque
         */
        const initActiveItemImage = () => {
          if (scope.edit_resource && scope.images && Array.isArray(scope.images)) {
            for (const image of scope.images) {
              image.active = false;
            }
            if (scope.edit_resource.image && scope.edit_resource.image.length > 0) {
              let index = scope.images.findIndex(i => i.src === scope.edit_resource.image);
              if (index > -1) {
                scope.images[index].active = true;
              }
            }
          }
        };

        /**
         * Permet de faire défiler le carousel
         * Désactivé lorsque fonctionne le drag/drop
         */
        const addDragToScroll = () => {
          const carousel = getCarousel();
          if (carousel) {
            let isDown = false;
            let startX;
            let scrollLeft;

            carousel.addEventListener('mousedown', e => {
              isDown = true;
              e.preventDefault();
              carousel.classList.add('active');
              startX = e.pageX - carousel.offsetLeft;
              scrollLeft = carousel.scrollLeft;
            });
            carousel.addEventListener('mouseleave', () => {
              isDown = false;
              carousel.classList.remove('active');
            });
            carousel.addEventListener('mouseup', () => {
              isDown = false;
              carousel.classList.remove('active');
            });
            carousel.addEventListener('mousemove', e => {
              if (!isDown) {
                return;
              }
              e.preventDefault();
              const x = e.pageX - carousel.offsetLeft;
              const walk = x - startX;
              carousel.scrollLeft = scrollLeft - walk;
            });
          }
        };

        /**
         * Renvoie l'entier supérieur à la valeur la plus grande
         * de la propriété id d'un item dans le tableau scope.items
         * @return {number} entier supérieur au id le plus grand de scope.items
         */
        const getNextId = () => {
          let nextId = -1;
          for (const item of scope.items) {
            if (Number.isInteger(item.id) && nextId < item.id) {
              nextId = item.id;
            }
          }
          nextId = nextId > -1 ? (nextId + 1) : 0;
          return nextId;
        };

        /**
         * Renvoie l'entier supérieur à la valeur la plus grande
         * de la propriété index d'un item dans le tableau scope.items
         * @return {number} entier supérieur au index le plus grand de scope.items
         */
        const getNextIndex = () => {
          let indexMax = 0;
          for (const item of scope.items) {
            if (item.index > indexMax) {
              indexMax = item.index;
            }
          }
          return indexMax + 1;
        };

        /**
         * Récupère l'élément HTML correspondant au carousel conteneur des groupes/liens
         * @return {HTMLElement | null} carousel des groupes ou des liens
         * ou null si document.getElementById a échoué
         */
        const getCarousel = () => {
          const itemType = scope.getItemType(false);
          return homeFactory.getCarousel(itemType);
        };

        /**
         * Attache les évènements du drag/drop aux items
         */
        const addDragDropEvents = () => {
          const item = scope.getItemType(false, true);
          const items = document.querySelectorAll(item);
          for (const item of items) {
            // attach the dragstart event handler
            item.addEventListener('dragstart', dragStart, false);
            item.addEventListener('dragenter', dragEnter, false)
            item.addEventListener('dragover', dragOver, false);
            item.addEventListener('dragleave', dragLeave, false);
            item.addEventListener('drop', drop, false);
          }
        };

        /**
         * Attache les évènements du drag/drop au dernier élément du tableau d'items
         * id du dernier élement ajouté
         */
        const addDragDropEventsLast = () => {
          const itemType = scope.getItemType(false);
          const index = scope.items.length - 1;
          const item = document.getElementById(itemType + '-' + index);
          if (item) {
            // attach the dragstart event handler
            item.addEventListener('dragstart', dragStart, false);
            item.addEventListener('dragenter', dragEnter, false)
            item.addEventListener('dragover', dragOver, false);
            item.addEventListener('dragleave', dragLeave, false);
            item.addEventListener('drop', drop, false);
          }
        };

        /**
         * Au clic pour déplacer,
         * enregistre l'id de l'item sélectionné dans les données de transfert
         * @param {object} e évènement dragStart, e.target est l'élément déplacé
         */
        const dragStart = (e) => {
          // selector = '.group' ou '.link'
          const selector = scope.getItemType(false, true);
          const item = e.target.closest(selector);
          if (item && item.id) {
            e.dataTransfer.setData('text/plain', item.id);
          }
        };

        /**
         * A l'entrée du survol d'un élément draggable
         * @param {object} e évènement dragEnter, e.target est l'élément survolé
         */
        const dragEnter = (e) => {
        };

        /**
         * Au mouvement de la souris lors du survol d'un élément draggable
         * @param {object} e évènement dragOver, e.target est l'élément survolé
         */
        const dragOver = (e) => {
          if(e.preventDefault) {
            e.preventDefault();
          }
        };

        /**
         * A la sortie du survol d'un élément draggable
         * @param {object} e évènement dragLeave, e.target est l'élément survolé
         */
        const dragLeave = (e) => {
        };

        /**
         * Au relaché de la souris pour déplacer l'item à la position actuellement survolée:<br>
         * Insère à la position survolée l'élément supprimé à la position de sélection.<br>
         * Met à jour la propriété index des items
         * @param {object} e évènement au relaché de la souris
         */
        const drop = (e) => {
          if(e.preventDefault) {
            e.preventDefault();
          }
          if(e.stopPropagation) {
            e.stopPropagation();
          }
          if (scope.mode === 'config') {
            // itemClass = '.group' ou '.link'
            const itemClass = scope.getItemType(false, true);

            // récupère l'élément draggable
            const id = e.dataTransfer.getData('text/plain');

            // récupère l'index d'origine dans scope.items
            const sourceIndex = getIndex(id);

            // récupère l'élément survolé destination:
            const target = e.target.closest(itemClass);
            // récupère l'index destination dans scope.items
            const targetIndex = getIndex(target.id);

            if (sourceIndex > -1 && targetIndex > -1) {
              // insère au rang target l'élément supprimé au rang source
              scope.items.splice(targetIndex, 0, scope.items.splice(sourceIndex, 1)[0]);
            }
            if (typeof scope.onOrderChange ==='function') {
              scope.onOrderChange(scope.items);
            }
          }
        };

        /**
         * Récupère le rang d'un item dans la collection d'enfants de l'élement HTML du carousel
         * @param {number} id string attribut id de l'élément dont on recherche le rang
         * @return {number} entier rang de l'item dans le carousel (i.e scope.items)
         */
        const getIndex = (id) => {
          let index = -1;
          const carousel = getCarousel();
          if (carousel && carousel.children.length > 0 && id !== '') {
            for (let i = 0; i < carousel.children.length; i++) {
              if (carousel.children[i].id === id) {
                index = i;
                break
              }
            }
          }
          return index;
        };

        /**
         * Transforme le nom du type d'objets gérés par le carousel 'group' ou 'link'
         * @param {boolean} capitalize est true pour mettre en majuscule le 1er caractère (pour les textes de locale)
         * @param {boolean} dotPrefix est true pour préfixer le nom avec un point (pour récupérer un item par classe css)
         * @return {string} le nom du type d'objet ('group' par exemple) après transformation
         */
        scope.getItemType = (capitalize, dotPrefix = false) => {
          let itemType = scope.isPrimary ? 'group' : 'link';
          if (capitalize) {
            itemType = gaJsUtils.capitalize(itemType);
          }
          if (dotPrefix) {
            itemType = '.' + itemType;
          }
          return itemType;
        };

        /**
         * Configure le défilement du carousel avec la molette de la souris
         */
        const addWheelScroll = () => {
          const carouselElement = getCarousel();
          if (carouselElement) {
            scope.whellScrollEnabled = true;
            carouselElement.addEventListener('wheel', (event) => {
              event.preventDefault();
              carouselElement.scrollBy({
                left: event.deltaY < 0 ? -30 : 30,
              });
            });
          }
        };

        /**
         * Renvoie l'image en string à partir du nom du fichier de l'image
         * Utilisé pour afficher les images (ng-style) dans la page d'accueil, les popovers et la popup
         * @param {string} filename nom du fichier de l'image
         * @return {string|null} image en string ou null
         */
        scope.getEncodedImageByName = (filename) => {
          if (scope.images && Array.isArray(scope.images) && scope.images.length > 0 && filename
              && filename.length > 0) {
            const image = scope.images.find(img => img.src === filename);
            if (image && image.hasOwnProperty('base')) {
              // renvoie la propriété 'base' de l'objet image
              // ayant 'src' égal au paramètre  'filename'
              return image.base;
            } else {
              const parentPath = 'img/admin/homepage/';
              return parentPath + (scope.isPrimary ? 'group/' : 'item/') + filename;
            }
          }
        };

        /**
         * Ouvre la page de l'application
         * Relance la récupération des applications si celles-ci ne sont pas initialisées.
         * @param {string} appuid identifiant de l'application (ex. "SIG", "Galilée"...)
         */
        scope.goToApp = (appuid) => {
          let app;
          if (appuid !== 'admin') {
            if (ApplicationFactory.resources.applications.length > 0) {
              if (appuid) {
                // récupère l'application dans les resources de ApplicationFactory
                app = ApplicationFactory.resources.applications.find(a => a.uid === appuid);
              }
              if (app !== undefined) {
                homeFactory.gotoApp(scope.portalId, app);
              } else {
                console.error(
                    $filter('translate')('portals.homepage.noApps') + ' - appuid = '
                    + appuid);
              }
            } else {
              // récupère les applications si les resources sont vides
              ApplicationFactory.get(true).then(
                  () => {
                    app = ApplicationFactory.resources.applications.find(a => a.name === appuid);
                    if (app !== undefined) {
                      // portalId est fourni en input du carousel de liens
                      homeFactory.gotoApp(scope.portalId, app);
                    } else {
                      console.error(
                          $filter('translate')('portals.homepage.noApps') + ' - appuid = '
                          + appuid);
                    }
                  },
                  () => {
                    console.error($filter('translate')('portals.homepage.getAppsError'));
                  }
              );
            }
          } else {
            // le carousel des groupes ne reçoit pas portalId: on récupère portalId dans le storage
            // si portalid est null goToApp renvoie quand même vers admin
            const portalid = PortalsFactory.getPortalId();
            homeFactory.gotoApp(portalid, {uid: 'admin'});
          }
        };

        /**
         * Ajoute l'image déposée dans la dropzone dans la mosaïque d'images
         * On ne devrait pas copier une image déposée dans le dossier CSS-HTML avant confirmation
         * mais les specs du tickets nécessitent ce comportement
         */
        scope.updateDropzoneImage = () => {
          if (scope.dzMemShare.dropzoneComponent.files.length > 0 && !scope.uploadPending) {
            // dépôt d'une image dans la dropzone

            // empêche complete-action d'être exécutée 2x par le composant dropzone
            scope.uploadPending = true;

            let filename = scope.dzMemShare.dropzoneComponent.files[0].name;
            const extension = filename.split('.').pop();
            filename = filename.split('.')[0] + (scope.isPrimary ? '-group.' : '-item.')
                + extension;
            // copie l'image du dossier "UPLOAD" dans le dossier "CSS-HTML"
            homeFactory.copyUploadedImgToCssRepo(
                scope.portalid,
                scope.dzMemShare.uploadProcessID,
                filename,
                true
            ).then(
                (img) => {
                  if (img.data && img.data.length > 0) {
                    // dans homepageNews on gère uniquement le logo et les arrière-plans
                    // on met à jour le tableau 'other'
                    for (const image of scope.images) {
                      image.active = false;
                    }

                    const prefix = 'data:image/' + (extension === 'svg' ? extension + '+xml'
                            : extension)
                        + ';base64,';
                    // insère l'image activée dans la mosaïque
                    const image = {
                      base: prefix + img.data,
                      src: filename,
                      active: true
                    };
                    if (scope.images.findIndex(i => i.src === filename) === -1) {
                      scope.images.push(image);
                    }
                    if (scope.edit_resource) {
                      scope.edit_resource.image = filename;
                    }
                  } else {
                    require('toastr').error(
                        $filter('translate')('portals.homepage.addHomepageImageError'));
                  }
                },
                () => {
                  require('toastr').error(
                      $filter('translate')('portals.homepage.addHomepageImageError'));
                }
            ).finally(() => {
              scope.uploadPending = false;
            });
          }
        };

        /**
         * Lance l'ouverture de la popup qui propose de supprimer une image de la page d'accueil
         * @param {number} indexToDelete rang de l'image dans le tableau d'images de la directive
         * @see homeFactory.ShowDialogToDeleteImage
         */
        scope.ShowDialogToDeleteImage = (indexToDelete) => {
          homeFactory.ShowDialogToDeleteImage(scope.portalId, scope.images, indexToDelete)
        }

        /**********************
         *   INITIALISATION   *
         **********************/

         scope.isUserAdmin = homeFactory.isUserAdmin();

        if (!scope.halign) {
          scope.halign = 'left';
        }
        if (!scope.isPrimary) {
          // récupère les applications auxquelles l'utilisateur a le droit d'accéder
          scope.getUserApplications();
        }

        if (scope.items) {

          if (scope.isPrimary) {
            // affiche une branche de type 'circle' aux items du carousel des groupes
            for (const item of scope.items) {
              item.isCirular = true;
            }
          }
          // Quand un texte d'item a un valign null,
          // on vérifie si le nom de l'image svg contient 'top', 'center' ou 'bottom'
          setValignFromImagesIfEmpty();

          // active les listeners pour défiler le carousel en glissant
          // inactif quand drag/drop actif
          addDragToScroll();
        }

        // Afin de limiter l'usage au mode édition,
        // drag/drop est géré dans un watcher sur la variable 'mode'
        // @see scope.$watch('mode')

        element.ready(() => {
          $timeout(() => {
            // active défilement du carousel avec molette souris
            addWheelScroll();
          }, 1000);
        });
      },
    };
  };

  homepageCarousel.$inject = [
    '$window',
    '$timeout',
    '$rootScope',
    'ConfigFactory',
    'ApplicationFactory',
    'FilesFactory',
    '$filter',
    '$location',
    'gaJsUtils',
    'homeFactory',
    'ngDialog',
    'gaDomUtils',
    'PortalsFactory'
  ];
  return homepageCarousel;
});