'use strict';
define(() => {
  const wfsMonitor = (FeatureTypeFactory, processFactory, FeatureAttachmentFactory, $interval,
    $timeout, gaDomUtils) => {
    return {
      templateUrl:
          'js/XG/modules/model/views/directives/wfsMonitor.html',
      restrict: 'EA',
      scope: {
        popup: '=?',  // popup wfsMonitor servant uniquement à accéder à la méthode close() car la méthode globale closeThisDialog ne fonctionne pas ici.
        // Si une meilleure solution pour clore la popup est trouvée alors ce paramètre peut être supprimé
      },
      link: function (scope, elt) {

        /**
         * Avancement de la barre de progression d'une ligne du tableau
         * @param {string} componentName nom du composant en cours de mis à jour
         * @param {number} progress progression du processus
         */
        const updateProcessBar = (componentName, progress) => {
          const barId = 'progress-value-' + componentName.replace(' ', '');
          const progressBar = document.getElementById(barId);
          if (progressBar) {
            progressBar.style.width = `${progress}%`;
            // arrondit la barre de valeur pour que ses angles ne dépassent pas le conteneur
            if (progress > 97) {
              progressBar.style.borderBottomRightRadius = '5px';
              progressBar.style.borderTopRightRadius = '5px';
            } else {
              progressBar.style.borderBottomRightRadius = '0';
              progressBar.style.borderTopRightRadius = '0';
            }
          }
        };

        /**
         * Avancement de la barre de progression des mises à jour en cours
         */
        const updateProcessBars = () => {
          for (const wfsComponent of scope.wfsComponents) {
            if (wfsComponent.isUpdating) {
              updateProcessBar(wfsComponent.name, wfsComponent.process.progress);
            }
          }
        };

        /**
         * Dans le retour de la requête vérifiant l'état des processus de mise à jour WFS,
         * si aucun processus n'est encore en cours d'exécution,
         * alors la récurrence de la requête est annulée
         */
        const stopInterval = () => {

          // Stoppe l'exécution récurrente de la requête de la progression des processsus
          $interval.cancel(stop);

          // Bascule le bouton "Arrêter tout" vers "Mise à jour"
          scope.isWfsUpdating = false;

          // Envoie à FeatureTypeCtrl l'ordre d'arrêter la surveillance des processus
          scope.$emit('wfsUpdate');
        };

        /**
         * Ajout des eventListeners mouseenter/mouseleave sur les icônes "i" (info) de la table
         */
        const setPopupMinHeight = () => {
          const popupPanel = elt[0].closest('.popupPanel');
          popupPanel.style.minHeight = popupPanel.clientHeight + "px";
        };

        /**
         * Au clic sur la case à cocher de l'en-tête de la 1ère colonne,
         * coche la case de toutes les lignes de la table
         */
        scope.toggleSelectAllWfsComponents = () => {
          $timeout(() => {
            for (const wfsComponent of scope.wfsComponents) {
              wfsComponent.toUpdate = scope.allWfsProcessesSelected.value;
            }
          });
        };

        /**
         * Au clic sur une case à cocher d'une ligne,
         * vérifie si toutes les lignes sont cochées
         */
        scope.updateAllSelectedValue = () => {
          $timeout(() => {
            scope.allWfsProcessesSelected.value = scope.wfsComponents.every(comp => comp.toUpdate);
          });
        };

        /**
         * Action au clic sur le bouton "Mettre à jour"/"Arrêter tout".
         * Lance ou interrompt le processus de mise à jour du composant par WFS
         * @param {MouseEvent} event propagé au clic sur le bouton
         */
        scope.toggleAllWfsUpdate = (event) => {
          scope.isWfsUpdating ? stopWfsUpdates(event) : startWfsUpdates(event);
        };

        /***
         * Affiche les paramètres de connexion WFS du composant dont l'icône "i" est survolée
         * Affichage d'une tooltip au-dessus de l'icône.
         * @param {MouseEvent} event
         */
        scope.showTooltip = (event) => {
          // Icone survolée
          const iElement = event.target;

          if (iElement) {
            const tooltip = iElement.nextElementSibling;
            if (tooltip) {
              const popupPanel = iElement.closest('.popupPanel');
              if (popupPanel) {

                // Calcul du décalage à appliquer
                const popupRect = popupPanel.getBoundingClientRect();
                const iRect = iElement.getBoundingClientRect();
                tooltip.style.bottom = `calc(${popupRect.bottom - iRect.top}px + 0.5em)`
                tooltip.style.left = `${iRect.left + (iRect.width / 2) - popupRect.left}px`;
                tooltip.classList.remove('tooltip-hidden');
              }
            }
          }

        };

        /**
         * Masque les paramètres de connexion WFS du composant après survol de l'icône "i" dans la ligne du composant
         * @param {MouseEvent} event
         */
        scope.hideTooltip = (event) => {
          const iElement = event.target;
          if (iElement) {
            const tooltip = iElement.nextElementSibling;
            if (tooltip) {
              tooltip.classList.add('tooltip-hidden');
            }
          }
        };

        /**
         * Désactivation du bouton "Mise à jour"
         * @return {boolean} true si aucun composant n'est sélectionné
         */
        scope.disableUpdateButton = () => {
          return Array.isArray(scope.wfsComponents) && scope.wfsComponents.every(
            comp => !comp.toUpdate);
        };

        /**
         * Télécharge
         * - le fichier du rapport de mise à jour WFS situé dans ATTACHMENTS/{componentName}
         * - le fichier des objets qui n'ont pas pu être récupérés durant la mise à jour WFS situé dans ATTACHMENTS/{componentName}
         * @param {string} componentName nom du composant
         * @param {string} extension
         */
        scope.downloadReportOrCsv = (componentName, extension = '') => {
          const filename = extension === 'txt' ? `wfs_report_${componentName}.txt`
            : `wfs_warning_${componentName}.csv`;
          FeatureAttachmentFactory.getdownloadurl(filename, componentName).then(
            res => {
              if (res.data && res.data.length > 0) {
                const downloadUrl = escape(res.data);

                // Lance le téléchargement plutôt que d'afficher le fichier texte dans un onglet du navigateur
                const link = document.createElement('a');
                link.href = downloadUrl;
                link.download = filename;
                link.style.display = 'none';
                document.body.appendChild(link);
                link.click();
                document.body.removeChild(link);
              }
            },
            err => {
              console.error(err.data);
            }
          );
        };

        /**
         * Fermeture de la popup
         * (la méthode closeThisDialog ne fonctionne pas)
         */
        scope.close = () => {
          if (scope.popup) {
            scope.popup.close();
          }
        };

        /**
         * Construit un tableau à partir des composants ayant une mise à jour WFS (wfsConfig != null).
         * Créé par composant un objet ayant 3 propriétés:<ul><li>
         * <code>name</code> qui est le nom du composant (fti.name)</li><li>
         * <code>config</code> qui est la configuration WFS du composant (fti.wfsConfig)</li><li>
         * <code>process</code> qui est le processus de mise à jour WFS si un fichier est présent dans le dossier PROCESSUS</li></ul>
         */
        const getWfsComponentsAndProcesses = () => {

          const TABLE_CONTAINER_ID = '#wfsMonitor-body-container';
          gaDomUtils.showLocalLoader(TABLE_CONTAINER_ID);

          FeatureTypeFactory.findWfsFeatureTypes().then(
            res => {
              if (Array.isArray(res)) {
                const wfsComponents = res;
                if (wfsComponents.length === 0) {
                  scope.wfsComponents = [];
                  gaDomUtils.removeLocalLoader(TABLE_CONTAINER_ID);
                } else {
                  const componentNames = wfsComponents.map(comp => comp.name).join(',');
                  processFactory.getProcessesByNametype(componentNames, 'wfsUpdate').then(
                    res => {
                      if (res.data) {
                        for (const [componentName, process] of Object.entries(res.data)) {
                          const component = wfsComponents.find(comp => comp.name === componentName);
                          if (component) {
                            component.process = process;
                            component.isUpdating = process !== null && process.etat === 'RUNNING';
                          }
                        }
                      }
                      gaDomUtils.removeLocalLoader(TABLE_CONTAINER_ID);
                      scope.wfsComponents = wfsComponents;

                      scope.isWfsUpdating = scope.wfsComponents.some(comp => comp.isUpdating);

                      // récupération des processus tous les 750 ms
                      if (scope.isWfsUpdating) {
                        stop = $interval(() => {
                          getWfsProgression();
                        }, 750);
                      }

                      // Hauteur de la popup
                      $timeout(setPopupMinHeight);

                      // Progression des processus en cours
                      $timeout(updateProcessBars);
                    },
                    err => {
                      console.error(err.data);
                      gaDomUtils.removeLocalLoader(TABLE_CONTAINER_ID);
                      scope.wfsComponents = wfsComponents;
                    }
                  );
                }
              }
            },
            () => {
              scope.wfsComponents = [];
              gaDomUtils.removeLocalLoader(TABLE_CONTAINER_ID);

              // Hauteur de la popup
              $timeout(setPopupMinHeight);
            }
          );
        };

        /**
         * Au clic sur le bouton "Mise à jour"
         * Lance le processus de mise à jour WFS des composants sélectionnés
         * @param {MouseEvent} event propagé au clic sur le bouton
         */
        const startWfsUpdates = (event) => {

          // évite la propriété outline du bouton
          event.target.blur();

          // tableau des noms de composants sélectionnés
          const toUpdateComponents = scope.wfsComponents.filter(comp => comp.toUpdate).map(
            comp => comp.name).join(',');

          if (toUpdateComponents.length > 0) {

            // envoie au FeatureTypeCtrl l'ordre de mise en évidence du bouton
            scope.$emit('wfsUpdate');

            FeatureTypeFactory.startWfsUpdates(toUpdateComponents).then(
              (res) => {
                if (res.data && Object.keys(res.data).length > 0) {

                  // mise à jour des processus des composants
                  for (const [componentName, process] of Object.entries(res.data)) {
                    const wfsComponent = scope.wfsComponents.find(
                      comp => comp.name === componentName);
                    if (wfsComponent) {
                      wfsComponent.process = process;
                      wfsComponent.isUpdating = true;
                    }
                  }

                  // bascule l'affchage du bouton "Mise à jour" sur "Arrêter tout"
                  scope.isWfsUpdating = true;

                  // récupération des processus tous les 750 ms
                  stop = $interval(() => {
                    getWfsProgression();
                  }, 750);
                }
              },
              err => {
                console.error(err.data);
              }
            );
          }
        };

        /**
         * Récupération des processus de mise à jour WFS en cours d'exécution
         *
         */
        const getWfsProgression = () => {

          // Tableau des processus en cours d'exécution
          const updatingProcessesUids = scope.wfsComponents.filter(
            comp => comp.process && comp.process.etat === 'RUNNING').map(
            comp => comp.process.uid).join(',');

          if (updatingProcessesUids.length > 0) {
            FeatureTypeFactory.checkWfsUpdates(updatingProcessesUids).then(
              res => {
                if (res.data) {

                  // Mise à jour des processus
                  for (const process of Object.values(res.data)) {
                    const component = scope.wfsComponents.find(
                      comp => comp.name === process.file);
                    if (component) {
                      component.process = process;

                      // Progression du processus
                      if (component.process && Number.isInteger(component.process.progress)
                            && component.process.progress > 0) {
                        updateProcessBar(component.name, component.process.progress);
                      }

                      component.isUpdating = process !== null && process.etat !== undefined
                            && process.etat === 'RUNNING';

                      // Gestion du spinner de la ligne
                      if (component.isTogglingUpdate) {
                        component.isTogglingUpdate = process.etat !== 'RUNNING'
                      }
                    }
                  }

                  // stoppe la récupération des processus lorsque
                  const isProcessing = scope.wfsComponents.some(comp => comp.process !== null
                        && comp.process !== undefined && comp.process.etat !== undefined
                        && comp.process.etat === 'RUNNING');

                  if (!isProcessing) {
                    stopInterval();

                    // Masque le spinner de la colonne "Progression"
                    scope.isTogglingAllUpdates = false;

                  }
                }
              },
              err => {
                console.error(err.data);
                stopInterval()

                // Masque le spinner de la colonne "Progression"
                scope.isTogglingAllUpdates = false;
              }
            );
          } else {
            stopInterval();
          }
        };

        /**
         * Au clic sur le bouton "Arrêter tout",
         * interrompt les processus de mise à jour WFS des composants en cours d'exécution
         * @param {MouseEvent} event propagé au clic sur le bouton
         */
        const stopWfsUpdates = (event) => {

          event.target.blur();

          // Affiche le spinner de la colonne "Progression"
          scope.isTogglingAllUpdates = true;

          // Tableau des rocessus en cours d'exécution
          const toStopProcesses = scope.wfsComponents.filter(
            comp => comp.process && comp.process.etat === 'RUNNING').map(
            comp => comp.process.uid).join(',');

          if (toStopProcesses.length > 0) {
            FeatureTypeFactory.stopWfsUpdates(toStopProcesses).then(
              (res) => {

                // Mise à jour des processus des composants en cours d'exécution
                const processesMap = res.data;
                for (const process of Object.values(processesMap)) {
                  const wfsComponent = scope.wfsComponents.find(
                    comp => comp.name === process.file);
                  if (wfsComponent) {
                    wfsComponent.process = process;
                    wfsComponent.isUpdating = false;
                  }
                }
                // Interruption la récupération des processus en cours d'exécution
                // Bascule de l'affchage du bouton "Arrêter tout" sur "Mise à jour"
                stopInterval();

                // Masque le spinner de la colonne "Progression"
                scope.isTogglingAllUpdates = false;

              },
              err => {
                console.error(err.data);
                stopInterval();
                scope.isTogglingAllUpdates = false;
              }
            );
          }
        };

        /**
         * Interrompt un processus de mise à jour WFS d'un composant.
         * Le processus prend le statut "INTERRUPTED" ("annulé").
         * Le thread est interrompu.
         * @param {string} processUid identifiant du processus à interrompre
         */
        scope.stopWfsUpdate = (processUid) => {

          // affiche le spinner de la ligne
          const toStopComponent = scope.wfsComponents.find(comp => comp.process !== null
              && comp.process !== undefined && comp.process.uid === processUid);
          if (!toStopComponent) {
            return;
          }
          toStopComponent.isTogglingUpdate = true;

          FeatureTypeFactory.stopWfsUpdates(processUid).then(
            (res) => {
              if (res.data) {
                const processMap = res.data;
                if (Object.entries(processMap).length > 0) {
                  const stoppedProcess = Object.values(processMap)[0];
                  stoppedProcess.etat = 'INTERRUPTED';
                  toStopComponent.process = stoppedProcess;
                  toStopComponent.isUpdating = false;

                  // masque le spinner de la ligne
                  toStopComponent.isTogglingUpdate = false;
                }
              }
              // Bascule le bouton "Arrêter tout" vers "Mise à jour" si aucun processus n'est en cours d'exécution
              scope.isWfsUpdating = scope.wfsComponents.some(comp => comp.isUpdating);
            },
            err => {
              console.error(err.data);
              scope.isTogglingUpdate = false;
            }
          )
        };

        /*****************************************************
         * Stop the interval when the directive is destroyed *
         ****************************************************/
        elt.on('$destroy', () => {
          if (stop) {
            $interval.cancel(stop);
          }
        });

        /******************
         * INITIALISATION *
         ******************/
        scope.allWfsProcessesSelected = {value: false};
        scope.isWfsUpdating = false;
        let stop;
        getWfsComponentsAndProcesses();
      },
    };
  };

  wfsMonitor.$inject = ['FeatureTypeFactory', 'processFactory', 'FeatureAttachmentFactory',
    '$interval', '$timeout', 'gaDomUtils'];
  return wfsMonitor;
});
