'use strict';
define(['angular'], function (angular) {
  var mod = angular.module('ga_jsutils_service', []);
  const resources = {
    serverDate: {
      deltaDateForClient: 0,
    },
    customLoader: {
      lastCall: 0,
      lastSrc: ''
    },
    toastrInfos: [],
    lockedLayers: []
  };



  Math.easeInOutQuad = function (t, b, c, d) {
    t /= d / 2;
    if (t < 1) {
      return (c / 2) * t * t + b;
    }
    t--;
    return (-c / 2) * (t * (t - 2) - 1) + b;
  };

  // used in levenshtein
  function editDistance(s1, s2) {
    s1 = s1.toLowerCase();
    s2 = s2.toLowerCase();

    var costs = new Array();
    for (var i = 0; i <= s1.length; i++) {
      var lastValue = i;
      for (var j = 0; j <= s2.length; j++) {
        if (i == 0) {
          costs[j] = j;
        } else {
          if (j > 0) {
            var newValue = costs[j - 1];
            if (s1.charAt(i - 1) != s2.charAt(j - 1)) {
              newValue = Math.min(Math.min(newValue, lastValue), costs[j]) + 1;
            }
            costs[j - 1] = lastValue;
            lastValue = newValue;
          }
        }
      }
      if (i > 0) {
        costs[s2.length] = lastValue;
      }
    }
    return costs[s2.length];
  }

  // used in guid
  function s4() {
    return Math.floor((1 + Math.random()) * 0x10000)
      .toString(16)
      .substring(1);
  }

  function getMyCurrentDate() {
    var dDate = new Date();
    if (isNaN(resources.serverDate.deltaDateForClient)) {
      //-- Resource non récupéré à temps: on fait au mieux.
      return new Date(dDate.getTime());
    }
    else {
      //-- Resource récupérée à temps: on fait comme il faut.
      return new Date(
        dDate.getTime() - resources.serverDate.deltaDateForClient
      );
    }
  }

  const getTypeDeProprietaireHardCodedValue = (typeProprietaireKey, valueSeparator) => {
    let displayValue = '';
    switch (typeProprietaireKey) {
      case '1':
        displayValue = 'Entreprise' + valueSeparator;
        break;
      case '2':
        displayValue = 'Particulier' + valueSeparator;
        break;
      case '3':
        displayValue = 'Sociéte' + valueSeparator;
        break;
      case '99':
        displayValue = 'Autre' + valueSeparator;
        break;
    }
    return displayValue;
  };

  /**
   * merge two features array and remove duplicates
   * @param a1
   * @param a2
   * @returns {*}
   */
  const mergeTwoFeatureArrays = (a1, a2) => {
    for (const feature1 of a1) {
      let newFeature = true;
      for (const feature2 of a2) {
        if (feature1.id === feature2.id) {
          newFeature = false;
        }
      }
      if (newFeature) {
        a2.push(feature1);
      }
    }
    return a2;
  };

  /**
   * Utilise une regex pour évaluer une chaine de caractère.
   * Evite la propagation de l'évènement clavier si la touche pressée
   * n'est pas un nombre de 0 à 9 (ou un point qd <code>isDotAllowed = true</code>)
   * @param {KeyboardEvent} e à la pression de la touche clavier
   * @param {boolean} isDotAllowed est true dans le cas de nombre à décimales (champ double)
   * @return {boolean} <code>true</code> si la saisie clavier contient
   * du texte non convertible en nombre
   */
  const preventTextTyping = (e, isDotAllowed) => {
    e = e || $window.event;
    const charStr = e.key;
    // autorise les nombres de 0 à 9 et le point.
    const regex = isDotAllowed ? /^-?\d*\.?\d*$/ : /^[0-9]+$/;
    if (!charStr.match(regex) || charStr === ' ') {
      e.preventDefault();
      return true;
    }
    return false;
  };

  /**
   * Evalue le navigateur actuellement utilisé
   * @return {string} nom courant du navigateur en cours d'utilisation
   */
  const checkBrowser = () => {
    const userAgent = window.navigator.userAgent;
    const browsers = {chrome: /chrome/i, safari: /safari/i, firefox: /firefox/i, ie: /internet explorer/i};
    for(let key in browsers) {
      if (browsers[key].test(userAgent)) {
        return key;
      }
    }
    return 'unknown';
  };


  /**
   * Crée un message d'erreur HTML stylisé pour SweetAlert avec un préfixe d'erreur traduit
   * @param {string} message - Le message d'erreur à afficher
   * @param {function} $filter - Service Angular $filter pour les traductions
   * @returns {string} Contenu HTML formaté du message d'erreur
   * @example
   * // Retourne du HTML formaté avec le message d'erreur traduit
   * const errorHtml = swalHtmlErrorMessage('Fichier non trouvé', $filter);
   */
  const swalHtmlErrorMessage = (message, $filter) => {
    const errorContainerStyle = 'max-height: 450px; overflow-y: auto; '
                + 'padding: 10px; border: 1px solid #ddd; border-radius: 4px;'
                + 'text-align: left;';
    let htmlContent = '<div style="' + errorContainerStyle + '">'
      + $filter('translate')('itv.itvMarkInactive.processEndedError')
      + message.replace(/\n/g, '<br>') + '</div>';
    return htmlContent;
  };


  mod.provider('gaJsUtils', function() {
    this.$get = function(
      $filter,
      $location,
      $window,
      $http,
      $q,
      $timeout,
      FunctionFactory,
      ConfigFactory,
      FeatureTypeFactory,
      $rootScope
    ) {
      var listePays;
      var loadMapTimerPromise;
      var UID_PROPERTY_ = 'gc_uid__';
      var uidCounter_ = 0;

      function mapError(client) {
        if (client.status != 200) {
          require('toastr').error('Map: ' + client.statusText);
        } else {
          require('toastr').error('Map: ' + client.responseText);
        }
        loadMapTimerPromise = undefined;
      }

      /**
       *      Get server date in order to be able to adjust client date informmation
       *  so that it matches more the date of the server in case the device
       *  is not correctly set.
       */
      function getServerDate() {
        var deferred = $q.defer();
        var dDate = new Date();
        //-- Si on n'a pas encore obtenu la date du serveur
        //-- ou si cela fait longtemps qu'on l'a obtenue
        //-- on interroge le serveur.
        if (
          resources.serverDate.dateAsTime == undefined ||
            Math.abs(
              dDate.getTime() - resources.serverDate.serverTimeWhenGotten
            ) > 600000
        ) {
          FunctionFactory.execute(
            {parameters: []},
            'getServerDate',
            'ja'
          ).then(function (res) {
            dDate = new Date();
            resources.serverDate.serverTimeWhenGotten = parseInt(res.data);
            resources.serverDate.clientTimeWhenGotten = dDate.getTime();
            resources.serverDate.deltaDateForClient =
                dDate.getTime() - resources.serverDate.serverTimeWhenGotten;
            if (resources.serverDate.deltaDateForClient > 24000000) {
              //-- Si la date du périphérique est trop éloignée de celle du serveur
              //-- on prévient l'utilisateur et nous utiliserons celle du serveur.
              this.errorMessage('La date de votre périphérique est erronée !');
            }
            deferred.resolve('');
          });
        }
        return deferred.promise;
      }

      if (typeof cfgandroid == 'undefined') {
        getServerDate();
      }

      let increaseExtent = extent => {
        let extentSize = ol.extent.getSize(extent);
        //if extent size is too small we will have a rezolution 1:0
        if (extentSize && extentSize.length > 1
            && (extentSize[0] < 40 || extentSize[1] < 40)) {
          return increaseExtent(ol.extent.buffer(extent, 10));
        }
        return extent;
      };

      return {
        getUrlReverseGeocoder: function (coords, type) {
          var url;
          if (type === 'nominatim') {
            url =
              'https://nominatim.openstreetmap.org/reverse?format=json&lat=' +
              coords[1] +
              '&lon=' +
              coords[0] +
              '&zoom=18&addressdetails=1&place=state&addressdetails=1&namesdetails=1';
          }
          else if (type === 'ban') {
            url =
              'https://api-adresse.data.gouv.fr/reverse/?lon=' +
              coords[0] +
              '&lat=' +
              coords[1];
          }
          else if (type === 'google') {
            url =
                'https://maps.googleapis.com/maps/api/geocode/json?latlng=' +
                coords[1] +
                ',' +
                coords[0];
          }

          return url;
        },

        transformGeocoderRes: function (res, type) {
          var geometry = {
            type: 'Point',
            coordinates: [],
          };
          var properties = {};

          if (type !== 'google') {
            if (type === 'nominatim') {
              geometry.coordinates.push(res.data.lon);
              geometry.coordinates.push(res.data.lat);

              Object.keys(res.data.address).forEach(function (att) {
                if (
                  ~[
                    'city',
                    'region',
                    'island',
                    'town',
                    'village',
                    'hamlet',
                    'suburb',
                    'locality',
                    'farm',
                    'airport',
                    'house',
                    'house_name',
                    'house_number',
                    'moor',
                    'islet',
                    'houses',
                  ].indexOf(att)
                ) {
                  properties.city = res.data.address[att];
                }
                if (
                  ~[
                    'motorway_junction',
                    'motorway',
                    'trunk',
                    'primary',
                    'secondary',
                    'tertiary',
                    'residential',
                    'unclassified',
                    'living_street',
                    'service',
                    'track',
                    'road',
                    'byway',
                    'bridleway',
                    'cycleway',
                    'pedestrian',
                    'footway',
                    'steps',
                    'motorway_link',
                    'trunk_link',
                    'primary_link',
                  ].indexOf(att)
                ) {
                  properties.street = res.data.address[att];
                }
              });

              properties = {
                context: angular.isDefined(res.data.address.state)
                  ? res.data.address.state
                  : '',
                label: angular.isDefined(res.data.display_name)
                  ? res.data.display_name
                  : '',
                postcode: angular.isDefined(res.data.address.postcode)
                  ? res.data.address.postcode
                  : '',
                housenumber: '',
              };
            }
            else {
              return res;
            }
          }
          return {
            data: {
              type: 'FeatureCollection',
              features: [
                {
                  type: 'Feature',
                  geometry: geometry,
                  properties: properties,
                },
              ],
            },
          };
        },

        /**
         * capitalizeFirstLetter
         * @param string
         * @returns {string}
         */
        capitalizeFirstLetter: function (string) {
          return string.charAt(0).toUpperCase() + string.slice(1);
        },

        /**
         * arrayUnique
         * @param string
         * @returns {string}
         */
        arrayUnique: function (a) {
          if (!(a instanceof Array)) {
            return a;
          }
          return a.reduce(function (p, c) {
            if (p.indexOf(c) < 0) {
              p.push(c);
            }
            return p;
          }, []);
        },

        /**
         * Nettoie une feature de paramètres logiques ajoutés par le client
         * @param {} x
         */
        getCleanFeature: (obj) => {
          let tmp = angular.copy(obj);

          if (tmp.properties) {
            for (let i in tmp.properties) {
              if (
                typeof tmp.properties[i] == 'array' ||
                  typeof tmp.properties[i] == 'object'
              ) {
                delete tmp.properties[i];
              }
            }
          }

          if (typeof tmp.geometry == 'string') {
            tmp.geometry = JSON.parse(tmp.geometry);
          }

          return tmp;
        },

        /**
         * When provided with the a.b.c.d dotSeparatedString, return
         * - the property object[a][b][c][d] if it exists
         * - false if not
         *
         * @param dotSeparatedString
         * @param {Object[]} [object]
         * @returns {*}
         */
        checkNestedProperty: function (dotSeparatedString, object) {
          if (!object) {
            object = false;
          }
          if (object == null) {
            return null;
          }
          return dotSeparatedString.split('.').reduce(function (obj, i) {
            if (obj == null) {
              return null;
            }
            return angular.isDefined(obj[i]) ? obj[i] : false;
          }, object);
        },

        /**
         * setNestedProperty
         * @param dotSeparatedString
         * @param obj
         * @param value
         */
        setNestedProperty: function (dotSeparatedString, obj, value) {
          var parts = dotSeparatedString.split('.'),
            part;
          var last = parts.pop();
          while ((part = parts.shift())) {
            if (typeof obj[part] != 'object') {
              obj[part] = {};
            }
            obj = obj[part];
          }
          if (angular.isDefined(value)) {
            obj[last] = value;
          }
        },
        createUuid: function(){
          let dt = new Date().getTime();
          const uuid = 'xxxxxxxx-xxxx-4xxx-yxxx-xxxxxxxxxxxx'.replace(/[xy]/g, (c) => {
            const r = (dt + Math.random()*16)%16 | 0;
            dt = Math.floor(dt/16);
            return (c=='x' ? r :(r&0x3|0x8)).toString(16);
          });
          return uuid;
        },


        /**
         * check if API return is parseable preventing error
         */
        isJsonParsable: function (string) {
          try {
            JSON.parse(string);
          } catch (e) {
            return false;
          }
          return true;
        },

        removeNestedProperty: (dotSeparatedString, obj) => {
          const parts = dotSeparatedString.split('.');
          let part;
          let last = parts.pop();
          while ((part = parts.shift())) {
            if (typeof obj[part] != 'object') {
              obj[part] = {};
            }
            obj = obj[part];
          }
          delete obj[last];
        },

        /**
         * sort array of objects by object key
         * @param array
         * @param key
         * @returns {*}
         */
        sortByKey: function (array, key) {
          return angular.isDefined(array)
            ? array.sort(function (a, b) {
              var x = a[key];
              var y = b[key];
              return x < y ? -1 : x > y ? 1 : 0;
            })
            : array;
        },

        /**
         * sort array of features by property
         * @param array
         * @param property
         * @returns {*}
         */
        sortByFeatureProperty: function (array, property) {
          return array.sort(function (a, b) {
            var x = a.properties[property];
            var y = b.properties[property];
            return x < y ? -1 : x > y ? 1 : 0;
          });
        },

        /**
         * sort array of features ([feature,feature], not a geotools FeatureCollection) by key
         * @param array
         * @param key
         * @returns {*}
         */
        sortArrayOfFeaturesByKey: function (array, key, type) {
          return array.sort(function (a, b) {
            var x = a.properties[key];
            var y = b.properties[key];

            if (angular.isUndefined(x)) {
              x = type == 'number' ? 0 : '';
            }
            if (angular.isUndefined(y)) {
              y = type == 'number' ? 0 : '';
            }

            return x < y ? -1 : x > y ? 1 : 0;
          });
        },

        /**
         * specific sort for feature objects
         * @param array
         * @param orderby
         * @returns {*}
         */
        sortByFeatureId: function (array, orderby) {
          var sortedArray = array.sort(function (a, b) {
            var x = parseInt(a.id.split('.')[1]);
            var y = parseInt(b.id.split('.')[1]);

            return x < y ? -1 : x > y ? 1 : 0;
          });
          return orderby == 'asc' ? sortedArray : sortedArray.reverse();
        },

        sortDataByNameIgnoreCase: (data) =>{
          data.sort((a,b) => (a.name.toLowerCase() > b.name.toLowerCase()) ? 1
            : ((b.name.toLowerCase() > a.name.toLowerCase()) ? -1 : 0));
        },

        /**
         * generateShortUID
         * @param chars
         * 36^4  = 1,6 million possibilities
         * 36^6  = 2,2 billion possibilities
         */
        generateShortUID: function (chars) {
          if (!chars) {
            chars = 4;
          }
          return (
            '0000' + ((Math.random() * Math.pow(36, chars)) << 0).toString(36)
          ).slice(-chars);
        },

        /**
         * setNewFeatureCollection
         * Allow to create a new feature collection from an array of features
         * foreach feature, if "type" and "properties" are missing, they will be added
         * @param features
         * @returns {{type: string, totalFeatures: number, features: Array}}
         */
        setNewFeatureCollection: function (features) {
          var ftc = {
            type: 'FeatureCollection',
            totalFeatures: 0,
            features: [],
          };
          if (features) {
            if (!angular.isArray(features)) {
              features = [features];
            }
            features.forEach(function (f) {
              ftc.totalFeatures++;

              if (!Object.prototype.hasOwnProperty.call(f, 'properties')) {
                var props = {};
                for (var j in f) {
                  props[j] = f[j];
                  delete f[j];
                }
                f.properties = props;
              }
              if (!Object.prototype.hasOwnProperty.call(f, 'type')) {
                f.type = 'Feature';
              }

              ftc.features.push(f);
            });
          }
          return ftc;
        },

        /**
         * Plain javascript equivalent of jquery offset()
         * @param elem
         * @returns {{top: number, left: number}}
         */
        getCoords: function (elem, parent) {
          if (!angular.isDefined(elem)) {
            return {top: 0, left: 0};
          }

          var box = elem.getBoundingClientRect();

          var body = document.body;
          var docEl = document.documentElement;

          var scrollTop =
              window.pageYOffset || docEl.scrollTop || body.scrollTop;
          var scrollLeft =
              window.pageXOffset || docEl.scrollLeft || body.scrollLeft;

          var clientTop = docEl.clientTop || body.clientTop || 0;
          var clientLeft = docEl.clientLeft || body.clientLeft || 0;

          var top = box.top + scrollTop - clientTop;
          var left = box.left + scrollLeft - clientLeft;

          // relative to a parent element, not window
          if (parent) {
            var parentbox = parent.getBoundingClientRect();
            top -= parentbox.top;
            left -= parentbox.left;
          }

          return {top: Math.round(top), left: Math.round(left)};
        },

        /**
         * Retrieve the height needed to fill an element to the bottom of the page
         * when it can't be done via css (ex: datatable in some cases)
         * @param elem
         */
        getFillToBottomHeight: function (elem) {
          var offset = this.getCoords(elem);
          return window.innerHeight - offset.top;
        },

        /**
         * scrollToElement
         */
        scrollToElement: function (element, to, duration) {
          var start = element.scrollTop,
            change = this.getCoords(to, element).top,
            currentTime = 0,
            increment = 20;

          // scroll slightly above
          change -= 50;

          var animateScroll = function () {
            currentTime += increment;
            var val = Math.easeInOutQuad(currentTime, start, change, duration);
            if (val < 0) {
              val = 0;
            }
            element.scrollTop = val;
            if (currentTime < duration) {
              setTimeout(animateScroll, increment);
            }
          };
          animateScroll();
        },

        /**
         * cast Attribute Value Type
         * @param value
         * @param type
         * @param new_type
         * @returns {*}
         */
        castAttributeValueType: function(value, new_type) {
          if (new_type == 'java.lang.Integer') {
            value = parseInt(value);
          }
          else if (new_type == 'java.lang.Double') {
            value = parseFloat(value);
          }
          else if (new_type == 'java.lang.Boolean') {
            value = Boolean(value);
          }
          else if (
            new_type == 'java.sql.Timestamp' ||
              new_type == 'java.util.Date'
          ) {
            if (new Date(value).toString() !== 'Invalid Date' && value != '') {
              var date = moment(value);
              date = date.toISOString();
              date = $filter('date')(date, 'yyyy-MM-ddTHH:mm:ss.sssZ');
              value = date;
            }
            else {
              value = null;
            }
          }
          else {
            value = value + '';
          }
          return value;
        },

        /**
         * Open new window with dual screen support
         */
        openNewWindowDS: function (url, title, w, h) {
          var dualScreenLeft =
              window.screenLeft != undefined ? window.screenLeft : screen.left;
          var dualScreenTop =
              window.screenTop != undefined ? window.screenTop : screen.top;

          var height = window.innerHeight
            ? window.innerHeight
            : document.documentElement.clientHeight
              ? document.documentElement.clientHeight
              : screen.height;

          var left = dualScreenLeft;
          var top = height / 2 - h / 2 + dualScreenTop;
          var newWindow = window.open(
            url,
            title,
            'directories=no,titlebar=no,toolbar=no,location=no,status=no,menubar=no,scrollbars=no,resizable=no, width='
              +
              w +
              ', height=' +
              h +
              ', top=' +
              top +
              ', left=' +
              left
          );

          // Puts focus on the newWindow
          if (window.focus) {
            newWindow.focus();
          }
        },

        /**
         * is Object Value Equal
         * @param a
         * @param b
         * @returns {boolean}
         */
        isObjectValueEqual: function (a, b) {
          // Of course, we can do it use for in
          // Create arrays of property names
          var aProps = Object.getOwnPropertyNames(a);
          var bProps = Object.getOwnPropertyNames(b);

          // If number of properties is different,
          // objects are not equivalent
          if (aProps.length != bProps.length) {
            return false;
          }

          for (var i = 0; i < aProps.length; i++) {
            var propName = aProps[i];

            // If values of same property are not equal,
            // objects are not equivalent
            if (a[propName] !== b[propName]) {
              return false;
            }
          }

          // If we made it this far, objects
          // are considered equivalent
          return true;
        },

        /**
         * getFileExtension
         * @param fileName
         * @returns {*}
         */
        getFileExtension: function (fileName) {
          var extension = fileName.match(/\.[0-9a-z]+$/i);
          return extension ? extension[0] : '';
        },

        /**
         * getSimpleAttributeType
         * @param {*} attr
         */
        getSimpleAttributeType: function (attr) {
          var fieldType = '';

          switch (attr.type) {
            case 'java.util.Date':
            case 'java.sql.Timestamp':
            case 'java.sql.Time':
            case 'java.sql.Date':
            case 'java.lang.Timestamp':
              fieldType = 'date';
              break;
            case 'java.lang.Double':
            case 'java.lang.Float':
            case 'java.math.BigDecimal':
            case 'java.lang.Integer':
              fieldType = 'number';
              break;
            case 'java.lang.Boolean':
              fieldType = 'boolean';
              break;
            case 'g2c.hyperlink':
              fieldType = 'hyperlink';
              break;
            // extensions kis
            case 'ObjectsArray':
              fieldType = 'ObjectsArray';
              break;
            default:
              fieldType = 'string';
              break;
          }

          return fieldType;
        },

        /**
         * Algo distance entre deux chaines
         */
        levenshtein: function (s1, s2) {
          var longer = s1;
          var shorter = s2;
          if (s1.length < s2.length) {
            longer = s2;
            shorter = s1;
          }
          var longerLength = longer.length;
          if (longerLength == 0) {
            return 1.0;
          }
          return (
            (longerLength - editDistance(longer, shorter)) /
              parseFloat(longerLength)
          );
        },

        /**
         * generate almost guid
         */
        guid: function () {
          return (
            s4() +
              s4() +
              '-' +
              s4() +
              '-' +
              s4() +
              '-' +
              s4() +
              '-' +
              s4() +
              s4() +
              s4()
          );
        },

        /**
         * slugify string
         */
        slugify: function (value) {
          var rExps = [
            {re: /[\xC0-\xC6]/g, ch: 'A'},
            {re: /[\xE0-\xE6]/g, ch: 'a'},
            {re: /[\xC8-\xCB]/g, ch: 'E'},
            {re: /[\xE8-\xEB]/g, ch: 'e'},
            {re: /[\xCC-\xCF]/g, ch: 'I'},
            {re: /[\xEC-\xEF]/g, ch: 'i'},
            {re: /[\xD2-\xD6]/g, ch: 'O'},
            {re: /[\xF2-\xF6]/g, ch: 'o'},
            {re: /[\xD9-\xDC]/g, ch: 'U'},
            {re: /[\xF9-\xFC]/g, ch: 'u'},
            {re: /[\xC7-\xE7]/g, ch: 'c'},
            {re: /[\xD1]/g, ch: 'N'},
            {re: /[\xF1]/g, ch: 'n'},
          ];

          for (var i = 0, len = rExps.length; i < len; i++) {
            value = value.replace(rExps[i].re, rExps[i].ch);
          }

          return value
            .toLowerCase()
            .replace(/[^a-z0-9-]/g, ' ')
            .replace(/\-{2,}/g, '-');
        },

        /**
         * Returns whether str ends with suffix
         * @param {*} str
         * @param {*} suffix
         */
        endsWith: function (str, suffix) {
          return str.indexOf(suffix, str.length - suffix.length) !== -1;
        },

        getListePays: function () {
          var defer = $q.defer();
          if (typeof ancAppAndroid !== 'undefined') {
            var defer = $q.defer();
            defer.resolve(ancAppAndroid.getListePays(Math.random()));
            return defer.promise;
          }
          else{
            if (listePays != undefined) {
              defer.resolve(listePays);
            }
            else {
              $http
                .get('/services/{portalid}/config/getListePays')
                .then(function (res) {
                  listePays = res.data.liste;
                  defer.resolve(listePays);
                });
            }
            return defer.promise;
          }
        },

        /**
         * get local date fomat
         * @returns {*|string}
         */
        getLocaleDateString: function () {
          var formats = {
            'ar-SA': 'dd/MM/yy',
            'bg-BG': 'dd.M.yyyy',
            'ca-ES': 'dd/MM/yyyy',
            'zh-TW': 'yyyy/M/d',
            'cs-CZ': 'd.M.yyyy',
            'da-DK': 'dd-MM-yyyy',
            'de-DE': 'dd.MM.yyyy',
            'el-GR': 'd/M/yyyy',
            'en-US': 'M/d/yyyy',
            'fi-FI': 'd.M.yyyy',
            'fr-FR': 'dd/MM/yyyy',
            'he-IL': 'dd/MM/yyyy',
            'hu-HU': 'yyyy. MM. dd.',
            'is-IS': 'd.M.yyyy',
            'it-IT': 'dd/MM/yyyy',
            'ja-JP': 'yyyy/MM/dd',
            'ko-KR': 'yyyy-MM-dd',
            'nl-NL': 'd-M-yyyy',
            'nb-NO': 'dd.MM.yyyy',
            'pl-PL': 'yyyy-MM-dd',
            'pt-BR': 'd/M/yyyy',
            'ro-RO': 'dd.MM.yyyy',
            'ru-RU': 'dd.MM.yyyy',
            'hr-HR': 'd.M.yyyy',
            'sk-SK': 'd. M. yyyy',
            'sq-AL': 'yyyy-MM-dd',
            'sv-SE': 'yyyy-MM-dd',
            'th-TH': 'd/M/yyyy',
            'tr-TR': 'dd.MM.yyyy',
            'ur-PK': 'dd/MM/yyyy',
            'id-ID': 'dd/MM/yyyy',
            'uk-UA': 'dd.MM.yyyy',
            'be-BY': 'dd.MM.yyyy',
            'sl-SI': 'd.M.yyyy',
            'et-EE': 'd.MM.yyyy',
            'lv-LV': 'yyyy.MM.dd.',
            'lt-LT': 'yyyy.MM.dd',
            'fa-IR': 'MM/dd/yyyy',
            'vi-VN': 'dd/MM/yyyy',
            'hy-AM': 'dd.MM.yyyy',
            'az-Latn-AZ': 'dd.MM.yyyy',
            'eu-ES': 'yyyy/MM/dd',
            'mk-MK': 'dd.MM.yyyy',
            'af-ZA': 'yyyy/MM/dd',
            'ka-GE': 'dd.MM.yyyy',
            'fo-FO': 'dd-MM-yyyy',
            'hi-IN': 'dd-MM-yyyy',
            'ms-MY': 'dd/MM/yyyy',
            'kk-KZ': 'dd.MM.yyyy',
            'ky-KG': 'dd.MM.yy',
            'sw-KE': 'M/d/yyyy',
            'uz-Latn-UZ': 'dd/MM yyyy',
            'tt-RU': 'dd.MM.yyyy',
            'pa-IN': 'dd-MM-yy',
            'gu-IN': 'dd-MM-yy',
            'ta-IN': 'dd-MM-yyyy',
            'te-IN': 'dd-MM-yy',
            'kn-IN': 'dd-MM-yy',
            'mr-IN': 'dd-MM-yyyy',
            'sa-IN': 'dd-MM-yyyy',
            'mn-MN': 'yy.MM.dd',
            'gl-ES': 'dd/MM/yy',
            'kok-IN': 'dd-MM-yyyy',
            'syr-SY': 'dd/MM/yyyy',
            'dv-MV': 'dd/MM/yy',
            'ar-IQ': 'dd/MM/yyyy',
            'zh-CN': 'yyyy/M/d',
            'de-CH': 'dd.MM.yyyy',
            'en-GB': 'dd/MM/yyyy',
            'es-MX': 'dd/MM/yyyy',
            'fr-BE': 'd/MM/yyyy',
            'it-CH': 'dd.MM.yyyy',
            'nl-BE': 'd/MM/yyyy',
            'nn-NO': 'dd.MM.yyyy',
            'pt-PT': 'dd-MM-yyyy',
            'sr-Latn-CS': 'd.M.yyyy',
            'sv-FI': 'd.M.yyyy',
            'az-Cyrl-AZ': 'dd.MM.yyyy',
            'ms-BN': 'dd/MM/yyyy',
            'uz-Cyrl-UZ': 'dd.MM.yyyy',
            'ar-EG': 'dd/MM/yyyy',
            'zh-HK': 'd/M/yyyy',
            'de-AT': 'dd.MM.yyyy',
            'en-AU': 'd/MM/yyyy',
            'es-ES': 'dd/MM/yyyy',
            'fr-CA': 'yyyy-MM-dd',
            'sr-Cyrl-CS': 'd.M.yyyy',
            'ar-LY': 'dd/MM/yyyy',
            'zh-SG': 'd/M/yyyy',
            'de-LU': 'dd.MM.yyyy',
            'en-CA': 'dd/MM/yyyy',
            'es-GT': 'dd/MM/yyyy',
            'fr-CH': 'dd.MM.yyyy',
            'ar-DZ': 'dd-MM-yyyy',
            'zh-MO': 'd/M/yyyy',
            'de-LI': 'dd.MM.yyyy',
            'en-NZ': 'd/MM/yyyy',
            'es-CR': 'dd/MM/yyyy',
            'fr-LU': 'dd/MM/yyyy',
            'ar-MA': 'dd-MM-yyyy',
            'en-IE': 'dd/MM/yyyy',
            'es-PA': 'MM/dd/yyyy',
            'fr-MC': 'dd/MM/yyyy',
            'ar-TN': 'dd-MM-yyyy',
            'en-ZA': 'yyyy/MM/dd',
            'es-DO': 'dd/MM/yyyy',
            'ar-OM': 'dd/MM/yyyy',
            'en-JM': 'dd/MM/yyyy',
            'es-VE': 'dd/MM/yyyy',
            'ar-YE': 'dd/MM/yyyy',
            'en-029': 'MM/dd/yyyy',
            'es-CO': 'dd/MM/yyyy',
            'ar-SY': 'dd/MM/yyyy',
            'en-BZ': 'dd/MM/yyyy',
            'es-PE': 'dd/MM/yyyy',
            'ar-JO': 'dd/MM/yyyy',
            'en-TT': 'dd/MM/yyyy',
            'es-AR': 'dd/MM/yyyy',
            'ar-LB': 'dd/MM/yyyy',
            'en-ZW': 'M/d/yyyy',
            'es-EC': 'dd/MM/yyyy',
            'ar-KW': 'dd/MM/yyyy',
            'en-PH': 'M/d/yyyy',
            'es-CL': 'dd-MM-yyyy',
            'ar-AE': 'dd/MM/yyyy',
            'es-UY': 'dd/MM/yyyy',
            'ar-BH': 'dd/MM/yyyy',
            'es-PY': 'dd/MM/yyyy',
            'ar-QA': 'dd/MM/yyyy',
            'es-BO': 'dd/MM/yyyy',
            'es-SV': 'dd/MM/yyyy',
            'es-HN': 'dd/MM/yyyy',
            'es-NI': 'dd/MM/yyyy',
            'es-PR': 'dd/MM/yyyy',
            'am-ET': 'd/M/yyyy',
            'tzm-Latn-DZ': 'dd-MM-yyyy',
            'iu-Latn-CA': 'd/MM/yyyy',
            'sma-NO': 'dd.MM.yyyy',
            'mn-Mong-CN': 'yyyy/M/d',
            'gd-GB': 'dd/MM/yyyy',
            'en-MY': 'd/M/yyyy',
            'prs-AF': 'dd/MM/yy',
            'bn-BD': 'dd-MM-yy',
            'wo-SN': 'dd/MM/yyyy',
            'rw-RW': 'M/d/yyyy',
            'qut-GT': 'dd/MM/yyyy',
            'sah-RU': 'MM.dd.yyyy',
            'gsw-FR': 'dd/MM/yyyy',
            'co-FR': 'dd/MM/yyyy',
            'oc-FR': 'dd/MM/yyyy',
            'mi-NZ': 'dd/MM/yyyy',
            'ga-IE': 'dd/MM/yyyy',
            'se-SE': 'yyyy-MM-dd',
            'br-FR': 'dd/MM/yyyy',
            'smn-FI': 'd.M.yyyy',
            'moh-CA': 'M/d/yyyy',
            'arn-CL': 'dd-MM-yyyy',
            'ii-CN': 'yyyy/M/d',
            'dsb-DE': 'd. M. yyyy',
            'ig-NG': 'd/M/yyyy',
            'kl-GL': 'dd-MM-yyyy',
            'lb-LU': 'dd/MM/yyyy',
            'ba-RU': 'dd.MM.yy',
            'nso-ZA': 'yyyy/MM/dd',
            'quz-BO': 'dd/MM/yyyy',
            'yo-NG': 'd/M/yyyy',
            'ha-Latn-NG': 'd/M/yyyy',
            'fil-PH': 'M/d/yyyy',
            'ps-AF': 'dd/MM/yy',
            'fy-NL': 'd-M-yyyy',
            'ne-NP': 'M/d/yyyy',
            'se-NO': 'dd.MM.yyyy',
            'iu-Cans-CA': 'd/M/yyyy',
            'sr-Latn-RS': 'd.M.yyyy',
            'si-LK': 'yyyy-MM-dd',
            'sr-Cyrl-RS': 'd.M.yyyy',
            'lo-LA': 'dd/MM/yyyy',
            'km-KH': 'yyyy-MM-dd',
            'cy-GB': 'dd/MM/yyyy',
            'bo-CN': 'yyyy/M/d',
            'sms-FI': 'd.M.yyyy',
            'as-IN': 'dd-MM-yyyy',
            'ml-IN': 'dd-MM-yy',
            'en-IN': 'dd-MM-yyyy',
            'or-IN': 'dd-MM-yy',
            'bn-IN': 'dd-MM-yy',
            'tk-TM': 'dd.MM.yy',
            'bs-Latn-BA': 'd.M.yyyy',
            'mt-MT': 'dd/MM/yyyy',
            'sr-Cyrl-ME': 'd.M.yyyy',
            'se-FI': 'd.M.yyyy',
            'zu-ZA': 'yyyy/MM/dd',
            'xh-ZA': 'yyyy/MM/dd',
            'tn-ZA': 'yyyy/MM/dd',
            'hsb-DE': 'd. M. yyyy',
            'bs-Cyrl-BA': 'd.M.yyyy',
            'tg-Cyrl-TJ': 'dd.MM.yy',
            'sr-Latn-BA': 'd.M.yyyy',
            'smj-NO': 'dd.MM.yyyy',
            'rm-CH': 'dd/MM/yyyy',
            'smj-SE': 'yyyy-MM-dd',
            'quz-EC': 'dd/MM/yyyy',
            'quz-PE': 'dd/MM/yyyy',
            'hr-BA': 'd.M.yyyy.',
            'sr-Latn-ME': 'd.M.yyyy',
            'sma-SE': 'yyyy-MM-dd',
            'en-SG': 'd/M/yyyy',
            'ug-CN': 'yyyy-M-d',
            'sr-Cyrl-BA': 'd.M.yyyy',
            'es-US': 'M/d/yyyy',
          };
          return formats[navigator.language] || 'dd/MM/yyyy';
        },

        filterFunc: function (elements, name) {
          var b = false;
          if (elements && name) {
            if (name.toLowerCase().score(elements.toLowerCase()) > 0.1) {
              b = true;
            }
          } else {
            b = true;
          }
          return b;
        },

        /**
         * Get diff time in days
         * Param only date.getTime()
         */
        dayDiff: function (d1, d2) {
          d1 = d1 / 86400000;
          d2 = d2 / 86400000;
          return new Number(Math.ceil(d2 - d1)).toFixed(0);
        },

        addDays: function (date, days) {
          var dateToReturn = new Date(date);
          dateToReturn.addDays(days);
          return dateToReturn;
        },

        removeEmpty: function (originalObj, sortArray) {
          var obj = angular.copy(originalObj);

          var cleanNullOrEmpty = function (obj) {
            if (angular.isDefined(obj) && obj != null) {
              Object.keys(obj).forEach(function (key) {
                if (obj[key] && typeof obj[key] === 'object') {
                  cleanNullOrEmpty(obj[key]);
                }

                if (
                  obj[key] == null ||
                    obj[key] == '' ||
                    JSON.stringify(obj[key]) == '{}'
                ) {
                  delete obj[key];
                }

                if (Array.isArray(obj[key])) {
                  obj[key] = obj[key].sort();
                }
              });
            }

            return obj;
          };

          obj = cleanNullOrEmpty(obj, sortArray);

          return obj;
        },

        formatDate: function (dateSource) {
          var date;
          if (dateSource) {
            date = moment(dateSource);
            date = date.toISOString();
            date = $filter('date')(date, 'yyyy-MM-ddTHH:mm:ss.sssZ');
          }
          return date;
        },

        moveElementInArray: function (theArray, theIndex, direction) {
          if (
            (theIndex === 0 && direction === 'up') ||
              (theIndex === theArray.length - 1 && direction === 'down')
          ) {
            return false;
          }
          var newIndex = direction === 'up' ? theIndex - 1 : theIndex + 1;
          theArray.splice(theIndex, 0, theArray.splice(newIndex, 1)[0]);
        },

        checkCustomRegex: function (stringValue) {
          let regex = /^[a-zA-Z0-9\\s\\_\\.\\,\\@\\|\\+\\<\\>\\^\\;\\:\\'\\"\\$\\-]*$/;
          if (regex.test(stringValue)) {
            return true;
          }
          else {
            return false;
          }
        },

        getMap: function(aScope) {
          if (aScope == null) return undefined;
          if (aScope.map != undefined) return aScope.map;
          else return this.getMap(aScope.$parent);
        },

        errorMessage: function (errorMsg1, errorMsg2) {
          var errorMsg = '<h4>Erreur</h4> ';
          if (errorMsg2 != undefined && errorMsg2 != '') {
            errorMsg += '<br/><h4>Details</h4>' + errorMsg1;
            errorMsg += '<br/>' + errorMsg2;
          } else {
            errorMsg = errorMsg1;
          }
          require('toastr').error(errorMsg);
        },

        successMessage: function (successMsg) {
          require('toastr').success(successMsg);
        },

        /**
         *     Transformation de la requete HTTP enregquete POST.
         * -1- Lecture des paramétres de l'URL pour les écrires
         *     dans la liste des paramètres de typePOST.
         * -2- Envoi de la requête en POST
         * -3- Récupération et mise en place de l'image résultante
         *
         * @param {[[type]]} tile [[Description]]
         * @param {[[type]]} src [[Description]]
         */
        customLoader: function (tile, src) {
          function getWsFromUrl(url) {
            var ws, iPos;
            ws = url.replace('/geoserver/', '');
            iPos = ws.indexOf('/');
            if (iPos==0) {
              iPos = url.indexOf('/services');
              ws = url.substr(iPos+10);
              iPos = ws.indexOf('/');
              //ws = ws.substr(0,iPos);
            }
            return ws.substr(0, iPos);
          }

          const client = new XMLHttpRequest();

          /**
           * Enléve un paramétre s'il existe de la liste de paramétre "WMS"
           * destinés à GeoServer et listés dans "srcUrl".
           *
           * @param {*} paramName : Nom du paramétre à enlever
           * @param {*} srcUrl : Liste des paramétres de la requête HTTP
           * @returns
           */
          const rmParamFromSrc = (paramName, srcUrl) => {
            const ind1 = srcUrl.indexOf('&'+paramName+'=');
            if (ind1 !== -1) {
              const ind2 = srcUrl.indexOf('&', ind1 + 1);
              return srcUrl.substr(0, ind1) + srcUrl.substr(ind2);
            }
            return srcUrl;
          };


          /**
           * Enlever des paramétres quine changent rien à la demande faite
           * à GeoServer de fabriquer une carte.
           *
           * @returns : liste de paramétré sans les paramétres "t"
           *            et "gcqlfilter"
           */
          const getSrcWithoutSomeParameters = () => {
            //-- Filtre géographique vide sans intérêt dans la comparaison, et
            //-- comme il n'apparaît pas tout le temps, quand il est vide
            //-- cela peut être la cause d'appel en double d'un requête
            //-- à GeoServer.
            src = src.replace('&gcql_filter=&', '&');
            src = rmParamFromSrc('t', src);
            src = rmParamFromSrc('layerDefs', src);
            return src;
          };


          /**
           * Vérifie si la requête précdente de quelques milissecondes est
           * la même que la requête courante.
           * Si c'est la même, on stocke le tile à actualiser et on retourne
           * FAUX pour ne pas demander une carte supplémentaire à GeoServer.
           * Si ce n'est pas la même, ou si la requête précédente est ancienne,
           * on enegistre les caractéristiquyes de celle-ci pour la comparer
           * à la suivante, et on retourne VRAI, pour demander la requête
           * à GeoServer.
           *
           * @param {*} resCustomLoader : information des requêtes
           *                            récemment envoyées à GeoServer.
           * @returns VRAI s'il faut envoyer la requête à geoServer, FAUX sinon.
           */
          const checkGeoserverRequest = (resCustomLoader) => {

            //console.log('delat miliseconds: ' + (nowMilliSeconds - resCustomLoader.lastCall));

            const simpleSrc = getSrcWithoutSomeParameters(src);
            if (nowMilliSeconds - resCustomLoader.lastCall > 1000) {
              //-- Vidage du contenu des informations de travail
              //-- qui évite d'envoyer des requêtes inutiles à GeoServer.
              resCustomLoader.sameRequests = {};
            }
            if (nowMilliSeconds - resCustomLoader.lastCall < 100
              && simpleSrc === resCustomLoader.lastSrc) {
              //-- Même requête que la précédente dans un délai trés bref
              if (!resCustomLoader.sameRequests['' + resCustomLoader.lastCall]) {
                //-- Première même requête que la précédente
                //-- => initialiser la liste des images à renseigner
                resCustomLoader.sameRequests['' + resCustomLoader.lastCall]
                = { tiles: [] };
              }
              //-- Ajouter l'image à renseigner dans la liste qui sera utilisée
              //-- lorsque GeoServer aura retourné la carte demandée
              const lastCall
              = resCustomLoader.sameRequests['' + resCustomLoader.lastCall];
              lastCall.tiles.push(tile);
              console.log('rejected duplicated request');
              return false;
            }

            //-- Préparation des informations pour le cas où d'autres requêtes
            //-- demanderont la même carte à GeoServer dans la foulée
            resources.customLoader.lastTile = [];
            resources.customLoader.theTileMaybe = tile;
            resources.customLoader.lastCall = nowMilliSeconds;
            resources.customLoader.lastSrc = simpleSrc;
            return true;
          };


          /**
           * Lit l'image de la carte retournée par GeoServer.
           * Met en place l'image de la carte dans OpenLayers.
           *
           * @param {object} tile Le récipiendaire OpenLayers
           *                de l'image de la carte
           * @param {object} resCustomLoader Informations de gestion de requêtes
           *            demandant la même carte à geoServer et contenant
           *            des récipiendaires similaire à abreuver
           *            avec la même carte GeoServer.
           * @param {FileReader} reader Lecteur pour le résultat de la requête HTTP
           *            envoyée à geoServer
           * @param {number} nowMilliSeconds timestamp
           */
          const putMapImageInOpenLayers = (tile, resCustomLoader, reader,
            nowMilliSeconds) => {
            let urlResult= reader.result;
            if(urlResult.indexOf('data:') > -1 && urlResult.length === 5) {
              urlResult = reader.result.replace('data:','');
            }
            const previous
              = resCustomLoader.sameRequests['' + nowMilliSeconds];
            if (previous) {
              //-- Autres récipiendaires de la même carte
              for (const aTile of previous.tiles) {
                aTile.getImage().src = urlResult;
              }
            }
            //-- Récipiendaire principal (ou premier pour la même requête)
            tile.getImage().src = urlResult;
          };


          const nowMilliSeconds = new Date().getTime();
          const resCustomLoader = resources.customLoader;
          if (!checkGeoserverRequest(resCustomLoader)) {
            return;
          }

          /**
           * Affiche un toastr dans le cas ou il n'est pas déjà dans la liste
           * @param type {string} type du toastr: error, warning ou success
           * @param text {string} text du toastr
           */
          const displayToastrWithoutDupplicate = (type, text) => {
            if(!resources.toastrInfos.find(el => el === text)) {
              // On ajoute notre texte à toastrInfos, les toastr ayant le meme texte ne seront pas affichés
              resources.toastrInfos.push(text);
              require('toastr')[type](text);
              $timeout(() => {
                resources.toastrInfos = resources.toastrInfos.filter(el => el !== text);
              }, 6000);
            }
          }

          //-- GET ALL THE PARAMETERS OUT OF THE SOURCE URL
          const dataEntries = src.split('&');
          let url;
          let params = '';
          let ws,
            doPost = true;
          for (let i = 0; i < dataEntries.length; i++) {
            if (i === 0) {
              url = dataEntries[i];
              ws = getWsFromUrl(url);
              if (url.indexOf('esriproxy') != -1) {
                doPost = false;
                if (src.substr(0, 1) != '/') {
                  src = '/' + src;
                }
                break;
              }
            } else {
              if (dataEntries[i].substr(0, 7).toLowerCase() === 'layers=') {
                dataEntries[i] =
                    dataEntries[i].substr(0, 7) +
                    ws +
                    ':' +
                    dataEntries[i].substr(7);
                dataEntries[i] = dataEntries[i].replace(
                  /%2C/g,
                  '%2C' + ws + ':'
                );
              }
              params = params + '&' + dataEntries[i];
            }
          }

          if (doPost) {
            client.open('POST', window.location.origin + url, true);
          }
          else {
            client.open('GET', src);
          }
          client.responseType = 'blob';

          client.onloadend = () => {
            // On recoit l'image seule de geoserver
            if (client.status === 200) {

              // On reçoit une image
              if (client.response.type === 'image/png') {
                const reader = new FileReader();
                reader.onloadend = () => {
                  putMapImageInOpenLayers(tile, resCustomLoader, reader,
                    nowMilliSeconds);
                };
                reader.readAsDataURL(client.response);
              }

              // On recoit un texte JSON contenant image + messages d'erreur
              else if (client.response.type === 'application/json') {
                // Récupération du texte du blob
                client.response.text().then((response) => {
                  const jsonResponse = JSON.parse(response);

                  // Gestion de l'image
                  if (jsonResponse.image) {
                    // Creation du blob de l'image à partir de l'image encodée en base64 dans le JSON
                    const binaryString = atob(jsonResponse.image);
                    const uint8Array = new Uint8Array(binaryString.length);
                    for (let i = 0; i < binaryString.length; i++) {
                      uint8Array[i] = binaryString.charCodeAt(i);
                    }
                    const pictureBlob = new Blob([uint8Array],
                      { type: 'image/png' });
                    // Ajout de l'image à la carte
                    const reader = new FileReader();
                    reader.onloadend = () => {
                      putMapImageInOpenLayers(tile, resCustomLoader, reader,
                        nowMilliSeconds);
                    };
                    reader.readAsDataURL(pictureBlob);
                  }

                  // Gestion des messages d'erreur
                  if (jsonResponse.hasOwnProperty('errorMessages')
                      && Array.isArray(jsonResponse.errorMessages)) {
                    for (let errorMessage of jsonResponse.errorMessages) {

                      // Si on détecte une erreur dans un style
                      if (errorMessage && errorMessage.match('Style') !== null
                          && errorMessage.match(ConfigFactory.portalid) !== null) {
                        // Récupération de l'erreur
                        const xmlDoc = new DOMParser().parseFromString(errorMessage, 'application/xml');
                        const serviceException = xmlDoc.querySelector('ServiceException').textContent;

                        // Récupération du nom de la couche en erreur
                        const serviceExceptionSplit = serviceException.split(':');
                        if (serviceExceptionSplit.length === 3) {
                          const errorLayerName = serviceExceptionSplit[2];

                          // Toastr warning prévenant que la couche est en erreur
                          displayToastrWithoutDupplicate('warning', $filter('translate')('common.featuretypes.errorInStyle')
                              + errorLayerName + $filter('translate')('common.featuretypes.layerNotDisplayed'))
                        }
                      }
                    }
                  }
                });
              } else if (client.response.type.includes('text/xml')) {
                // KIS-3375: cas où un composant KIS est absent de Geoserver
                const reader = new FileReader();
                reader.onload = function(event) {
                  const textContent = event.target.result;
                  if (textContent.includes(':') && textContent.includes('Could not find layer')) {
                    const componentName = textContent.split(':').pop().trimEnd();
                    const fti = FeatureTypeFactory.resources.featuretypes.find(ft => ft.name === componentName);
                    let compAlias = componentName;
                    if (fti) {
                      compAlias = fti.alias
                      const error = $filter('translate')('common.featuretypes.layerAbsentInGeoserver').replace('$1', compAlias);
                      displayToastrWithoutDupplicate('error', error);
                      // KIS-3548: envoie un évènement à gcLayers pour que le tableau servant à construire le géocatalogue
                      // ajoute une propriété "locked" à la layer du composant problématique
                      if (!resources.lockedLayers.some(layerName => layerName === fti.name)) {
                        $rootScope.$broadcast('lockGeocatalogLayer', fti.name);
                        resources.lockedLayers.push(fti.name);
                      }
                    }
                  }
                };
                reader.readAsText(client.response, 'utf-8');
              } else {
                // KIS-3195: affichage de l'erreur issue d'un document XML
                console.error(client.response);
              }
            }
            // Cas d'erreur
            else {
              if (loadMapTimerPromise === undefined)
                loadMapTimerPromise = $timeout(function() {
                  mapError(client);
                }, 2500);
            }
          };

          client.setRequestHeader(
            'Content-type',
            'application/x-www-form-urlencoded'
          );
          //client.setRequestHeader("Content-length", params.length);
          if (doPost) {
            client.send(params);
          } else {
            client.send();
          }
        },

        /**
         *        Return current year as an integer value.
         *    An advantage of this function is that it gives the possibility to simulate
         *    another year in order to test the application.
         */
        getCurrentYear: function () {
          var dDate = getMyCurrentDate();
          return dDate.getFullYear();
        },

        /**
         *        Return current date as a date object.
         *    An advantage of this function is that it gives the possibility to simulate
         *    another day in order to test the application.
         */
        getCurrentDate: function (delta) {
          var dDate = getMyCurrentDate();
          dDate.setHours(0);
          dDate.setMinutes(0);
          dDate.setSeconds(0);
          if (delta != undefined) {
            dDate.setTime(dDate.getTime() + delta);
          }
          return dDate;
        },

        scaleToResolution: function(scale) {
          const isValidScale =
            scale != undefined &&
            ((typeof scale == 'string' && scale.length > 0) ||
              (typeof scale == 'number' && scale != 0));
          if (isValidScale) {
            const dpi = 25.4 / 0.28;
            return parseFloat(scale) / (39.37 * dpi);
          }
          else {
            return null;
          }
        },

        /**
         * Calcul de l'échelle en fonction de la résolution de la carte.
         *
         * @param {*} resolution Résolution de la carte
         * @param {*} units Unité pour laquelle l'échelle est calculée
         *                  (mètre ou pouce)
         * @returns Echelle de la carte
         */
        resolutionToScale: (resolution, units) => {
          const isValidRes =
          resolution != undefined &&
            ((typeof resolution == 'string' && resolution.length > 0) ||
              (typeof resolution == 'number' && resolution != 0));
          if (isValidRes) {
            // 1px = 0.28 mm according to the OGC spec for DPI
            const dpi = 25.4 / 0.28;
            units = units ? units : 'm';
            const mpu = ol.proj.METERS_PER_UNIT[units];
            return parseInt(
              (100 * parseFloat(resolution) * dpi) / (mpu * 2.54)
            );
          }
          else {
            return null;
          }
        },

        getAppName: function () {
          if ($location.search().app) {
            return $location.search().app;
          }
          else {
            return angular.module('gcMain').app;
          }
        },

        translateAttributeAlias: function (attr, fti) {
          var aliasTrans = $filter('translate')(
            'features.' + fti.name + '.properties.' + attr
          );
          if (aliasTrans.indexOf('features.') === -1) {
            return aliasTrans;
          }
          return attr.alias;
        },

        getUid: function (obj) {
          return (
            (Object.prototype.hasOwnProperty.call(obj, UID_PROPERTY_) &&
                  obj[UID_PROPERTY_]) ||
              (obj[UID_PROPERTY_] = ++uidCounter_)
          );
        },

        prepareThingsWhenNewSelection: function (scope, SelectManager, res) {
          if (scope.altKeyPressed) {
            if (SelectManager.getfeatures().features !== undefined) {
              for (
                let i = 0;
                i < SelectManager.getfeatures().features.length;
                i++
              ) {
                let arrToCompareFeatures = res.data.features.map(
                  feature => JSON.stringify(feature));

                if (
                  arrToCompareFeatures.indexOf(
                    JSON.stringify(SelectManager.getfeatures().features[i])) !== -1
                ) {
                  // remove selection if it is already there
                  res.data.features = res.data.features.filter(el => {
                    return el.id !== SelectManager.getfeatures().features[i].id;
                  });
                }
                else {
                  res.data.features.push(SelectManager.getfeatures().features[i]);
                }

              }
            }
            res.data.totalFeatures = res.data.features.length;
          }

          if (SelectManager.getpop()) {
            try {
              if (SelectManager.getpop()) {
                SelectManager.getpop().destroy();
              }
              if (SelectManager.getpop().scope) {
                SelectManager.getpop().scope.$broadcast('$destroy');
              }
            }
            catch (e) {
              SelectManager.setpop(null);
            }
          }
        },

        isNumeric: function (val) {
          let isNumeric = /^[-+]?(\d+|\d+\.\d*|\d*\.\d+)$/;
          return isNumeric.test(val);
        },

        downloadFile: function (httpPath, currentid, docName) {
          if (angular.isUndefined(currentid)) {
            currentid = '';
          }
          if (angular.isUndefined(docName)) {
            docName = 'Document';
          }
          $http({
            method: 'GET',
            url: httpPath,
            responseType: 'arraybuffer',
          }).then(function (response) {
            let filename = currentid + '_' + docName;
            let contentType = response.headers()['content-type'];
            let linkElement = document.createElement('a');
            try {
              var blob = new Blob([response.data], {type: contentType});
              var url = window.URL.createObjectURL(blob);
              linkElement.setAttribute('href', url);
              linkElement.setAttribute('download', filename);
              var clickEvent = new MouseEvent('click', {
                view: window,
                bubbles: true,
                cancelable: false,
              });
              linkElement.dispatchEvent(clickEvent);
            }
            catch (ex) {
              console.log(ex);
            }
          },
          function (response) {
            console.log(response);
          });
        },

        localiseGeom: function (geom, map) {
          var zoomHalfExtent = 25;
          var extent = geom.getExtent();
          if (extent[0] < 100) {
            //-- Cas où système de projection du style
            // -- de EPSG:4326 en degré, minutes, secondes.
            zoomHalfExtent = 0.0005;
          }
          //-- Appliquer une marge autour de l'étendue.
          extent[0] -= zoomHalfExtent;
          extent[2] += zoomHalfExtent;
          extent[1] -= zoomHalfExtent;
          extent[3] += zoomHalfExtent;
          extent = increaseExtent(extent);
          map.getView().fit(extent, map.getSize());
        },

        distanceBetweenPoints: function (p1, p2) {
          var a = p1[0] - p2[0];
          var b = p1[1] - p2[1];

          var dist = Math.sqrt(Math.pow(a, 2) + Math.pow(b, 2));

          return dist;
        },

        /**
         *    Récupération du menu contextuuel principal pour le passer
         * aux widgets ouverts depuis le menu de gauche.
         */
        getMenuOptions: function (pScope) {
          if (pScope.menuOptions) {
            return pScope.menuOptions;
          }
          if (pScope.$parent) {
            return this.getMenuOptions(pScope.$parent);
          }
          else {
            return undefined;
          }
        },

        errorListToOneString: function (errorList) {
          let mess = '',
            err;
          for (var iErr = 0; iErr < errorList.length; iErr++) {
            err = errorList[iErr];
            mess +=
                '\n' +
                err.composant +
                '.' +
                err.fonction +
                ': ' +
                err.exception +
                err.message_kis;
          }
          return mess;
        },

        firstOrDefault: (list, property, value, defaultValue) => {
          for (let index in list) {
            if (Object.prototype.hasOwnProperty.call(list[index], property)
                && list[index][property] === value) {
              return list[index];
            }
          }
          return defaultValue ? defaultValue : {};
        },

        isEmptyObject: object => {
          return Object.keys(object).length === 0 && object.constructor === Object;
        },

        getTextFromHtmlString: (text) => {
          if (typeof text === typeof Object()) {
            const htmlFromText = $(text.toString());
            if (htmlFromText && htmlFromText.length && htmlFromText.text) {
              return htmlFromText.text();
            }
          }
          return text;
        },

        getTypeDeProprietaireValue: (typeProprietaireKey, valueSeparator,
          typeProprietaireListeDeroulante) => {
          if (typeProprietaireListeDeroulante) {
            const value = typeProprietaireListeDeroulante[typeProprietaireKey];
            if (value) {
              return value;
            }
          }
          return getTypeDeProprietaireHardCodedValue(typeProprietaireKey, valueSeparator);
        },

        getZIndexMax: (map) => {
          let zIndex = 0;
          map.getLayers().forEach((layer) => {
            //If this is actually a group, we need to create an inner loop to go through its individual layers
            if (layer instanceof ol.layer.Group) {
              layer.getLayers().forEach((groupLayer) => {
                zIndex = groupLayer.getZIndex() > zIndex ? groupLayer.getZIndex() : zIndex;
              });
            } else {
              zIndex = layer.getZIndex() > zIndex ? layer.getZIndex() : zIndex;
            }
          });
          return zIndex;
        },

        mergeTwoFeatureArrays: (a1, a2) => {
          return mergeTwoFeatureArrays(a1, a2);
        },

        /**
         * Merge two result of the ogc/getFeat function in res1
         * @param res1
         * @param res2
         */
        mergeTwoGetFeatResult: (res1, res2) => {
          if (res2 && res2.features) {
            // features
            res1.features = mergeTwoFeatureArrays(res1.features, res2.features);

            // totalFeatures
            res1.totalFeatures = res1.features.length;

            // realTotalFeature
            const realTotalFeature = {};
            let featureName;
            for (const feature of res1.features) {
              featureName = feature.id.split('.');
              featureName.pop();
              featureName = featureName.join('');
              realTotalFeature[featureName] ?
                realTotalFeature[featureName]++ :
                realTotalFeature[featureName] = 1;
            }
            res1.realTotalFeatures = realTotalFeature;
          }
          return res1;
        },

        /**
         * Calcule l'identifiant dans le cas normal il s'ait de feature.id,
         * dansle acs ESRI avec indication d'un champ particulier
         * la valeur de ce champ constitue la partie numérique de l'identifiant.
         *
         * @param {object|ol.Feature} feature feature geojson ou OpenLayers
         * @param {object} fti FeatureTypeInfo du feature
         * @param {object} feat2 scope.result dans le cas de mise à jour
         *                    sous édition métier
         * @returns string au format ftiName.XXX
         */
        getIdInCaseEsriId: (feature, fti, feat2) => {
          let featProps = feature.properties;
          if (!featProps && feature.getProperties) {
            //-- Cas d'une feature OpenLayers.
            featProps = feature.getProperties();
          }
          if (fti.type === 'esri' && fti.esriIdField !== 'objectid') {
            if (featProps && featProps.hasOwnProperty(fti.esriIdField)) {
              return fti.name + '.' + featProps[fti.esriIdField];
            }
            else if (feat2 && feat2.hasOwnProperty(fti.esriIdField)) {
              return fti.name + '.' + feat2[fti.esriIdField];
            }
          }
          return feature.id;
        },

        simplifyMapModel: (mm) => {
          let iFti;
          for (iFti = 0; iFti < mm.data.length; iFti++) {
            if (mm.data[iFti].fti) {
              if (!mm.data[iFti].ftiUid) {
                mm.data[iFti].ftiUid = mm.data[iFti].fti.uid;
                mm.data[iFti].ftiName = mm.data[iFti].fti.name;
              }
              delete mm.data[iFti].fti;
            }
            else {
              return false;
            }
          }
          return true;
        },

        // Vérifie l'existence de propriétés imbriquées dans un objet ou un tableau d'objets
        // @param value est un objet ou un tableau d'objets
        // @param nested est une string ou un tableau de propriétés imbriquées à vérifier dans value
        // exemple: nested = 'properties.title.name' ou ['properties','title','name']
        // @param checkNestedOnFirstValue est false quand on doit tester
        // l'existence des propriétés imbriquées dans chaque élément
        // du tableau value (uniquement dans le cas où value est un tableau)
        // @return true si nested n'existe pas et value est non null et défini,
        // true si nested existe et quand value et les string de nested sont
        // des propriétés non null et définies de value
        // (ou seulement des propriétés non null et définies du 1er élément de value
        // dans le cas où checkNestedOnFirstValue est true et value un tableau)
        notNullAndDefined: (value, nested, checkNestedOnFirstValue) => {
          let notNullAndDefined = false;
          const checkNestedProperties = (value, children, notNullAndDefined) => {
            notNullAndDefined = !!notNullAndDefined;
            if (Array.isArray(value)) {
              console.log('jsUtils.notNullAndDefined - présence de tableaux' +
                  ' imbriqués dans un tableau! Veuillez simplifier le 1er argument \'value\'');
            }
            const valueIsObject = value instanceof Object && !Array.isArray(value);
            if (valueIsObject && children !== null && children !== undefined && Array.isArray(
              children) && notNullAndDefined) {
              let property = value;
              for (const child of children) {
                if (Object.prototype.hasOwnProperty.call(property, child)
                  && property[child] === null || property[child] === undefined) {
                  notNullAndDefined = false;
                  break;
                }
                else {
                  property = property[child];
                }
              }
            }
            return notNullAndDefined;
          };

          notNullAndDefined = value !== null && value !== undefined;
          if (Array.isArray(value) && notNullAndDefined) {
            if (nested !== null && nested !== undefined && (Array.isArray(nested) || typeof nested
                === 'string')) {
              // laisse la possibilité de fournir nested
              // en string avec séparateur '.' ou en tableau de string
              let propertiesArray = null;
              if (Array.isArray(nested)) {
                propertiesArray = nested;
              }
              else if (typeof nested === 'string') {
                propertiesArray = nested.split('.');
              }
              if (!checkNestedOnFirstValue) {
                // Uniquement quand value est un tableau, checkNestedOnAllValues est true
                // lorsque on souhaite vérifier la nullité des propriété imbriquées
                // pour tous les élements du tableau value
                for (const val of value) {
                  notNullAndDefined = checkNestedProperties(val, propertiesArray,
                    notNullAndDefined);
                  if (!notNullAndDefined) {
                    break;
                  }
                }
              }
              else {
                // quand value est un tableau et checkNestedOnAllValues est false,
                // on ne vérifie que les propriétés imbriquées de la 1ère variable du tableau value
                notNullAndDefined = checkNestedProperties(value[0], propertiesArray,
                  notNullAndDefined);
              }
            }
            else {
              // quand value est un tableau et nested n'existe pas
              for (const val of value) {
                if (val === null || val === undefined) {
                  notNullAndDefined = false;
                  break;
                }
              }
            }
          }
          else if (value !== null && value !== undefined &&
              (nested !== null && nested !== undefined && (Array.isArray(nested) || typeof nested
                  === 'string'))
              && notNullAndDefined) {
            // value n'est pas un tableau
            let propertiesArray = Array.isArray(nested) ? nested : nested.split('.');
            notNullAndDefined = checkNestedProperties(value, propertiesArray, notNullAndDefined);
          }
          return notNullAndDefined;
        },

        isVideoFile: (filename) => {
          return filename.endsWith('.mov') || filename.endsWith('.mpg') || filename.endsWith('.mp4')
              || filename.endsWith('.mpeg') || filename.endsWith('.avi')
              || filename.endsWith('.webm') || filename.endsWith('.mkv');
        },

        //Vérifie la présence d'un zip dans les fichiers chargés
        // @param files fichiers chargés dans un input-file
        // @return {boolean} true si aucun fichier de la liste n'est un zip
        isFilesUnzipped: (files) => {
          for (const file of files) {
            if ((file.name && file.name.split('.').length === 2
                    && file.name.split('.')[1].toLowerCase() === 'zip')
                || file.type === 'application/x-zip-compressed') {
              return false;
            }
          }
          return true;
        },

        // Check is the user has the role
        /**
         * Check is the user has the role
         * @param user: user Object
         * @param role: string of the role ex: 'admin'
         * @returns {boolean}
         */
        userHasRole: (user, role) => {
          // root has every roles
          if (user.login === 'root') {
            return true;
          }
          // Check in roles
          for(let userRole of user.roles) {
            if(role === userRole.name) {
              return true;
            }
          }
          // Check in groups
          for(let group of user.groups) {
            for(let userRole of group.roles) {
              if(role === userRole.name) {
                return true;
              }
            }
            // Check in groups
            for(let group of user.groups) {
              for(let userRole of group.roles) {
                if(role === userRole.name) {
                  return true;
                }
              }
            }
          }
          return false;
        },
        // Ouvre un nouvel onglet vers l'Url
        // @param url string de l'url de la page de redirection
        redirectTo: (url) => {
          $window.open(url, '_blank');
        },
        // Ferme les popover des boutons bs-popover qui sont dans la class firstClass
        // Si firstClass est null on ferme toutes les popovers
        closeCfgPopovers: (firstClass) => {
          const selector = firstClass ? '.' + firstClass + ' .popover' : '.popover';
          $(selector).remove();
        },
        // récupère la propriété css d'un élément
        getStyle: (e, styleName) => {
          let styleValue = '';
          if (document.defaultView && document.defaultView.getComputedStyle) {
            styleValue = document.defaultView.getComputedStyle(e, '').getPropertyValue(styleName);
          }
          else if (e.currentStyle) {
            styleName = styleName.replace(/\-(\w)/g, (strMatch, p1) =>{
              return p1.toUpperCase();
            });
            styleValue = e.currentStyle[styleName];
          }
          return styleValue;
        },
        // Transforme le 1er caractère d'une string en majuscule
        capitalize: (str) => {
          if(typeof str === 'string') {
            return str.replace(/^\w/, c => c.toUpperCase());
          }
          else {
            return '';
          }
        },
        sortHtmlCollectionById: (htmlCollection) => {
          if (Array.isArray(htmlCollection) && htmlCollection.length > 0) {
            // magically coerce into an array first
            const items = htmlCollection.slice();

            // Now we can sort it.  Sort alphabetically
            items.sort((a, b) => {
              return a.id.localeCompare(b.id);
            });

            // reatach the sorted elements
            for(const item of items) {
              // store the parent node so we can reatach the item
              const parent = item.parentNode;
              // detach it from wherever it is in the DOM
              const detatchedItem = parent.removeChild(item);
              // reatach it.  This works because we are itterating
              // over the items in the same order as they were re-
              // turned from being sorted.
              parent.appendChild(detatchedItem);
            }
          }
        },

        /**
         * Remplace les caractères accentués d'une string par les caractères non-accentués.
         * Remplace les points d'une string par un espace (sauf le dernier point)
         * @param {string} inputString string à évaluer
         * @return {string} chaîne de caractères dépourvue de caractères accentués et possédant un point au maximum
         */
        replaceSpecialChars: (inputString) => {
          let regex = /[éèàâêëîçïôöùûü]/g;

          // compte le nombre de points dans la chaîne de caractères
          let count = (inputString.match(/\./g) || []).length;

          if (regex.test(inputString) || count > 1) {
            let validString = inputString;
            if (regex.test(inputString)) {
              // INPUT: "La chaîne de caractères avec des caractères spéciaux éèà"
              // OUTPUT: "La chaine de caracteres avec des caracteres speciaux eea"
              validString = validString.normalize('NFD').replace(/[\u0300-\u036f]/g, '');
            }
            if (count > 1) {
              // Si la chaîne de caractères contient plusieurs points
              // Remplace tous les points sauf le dernier par un espace
              validString = validString.replace(/\.(?=[^.]*\.)/g, ' ');
            }
            return validString;
          }
          return inputString;
        },

        /**
         * Remplace les caractères accentués par les caractères non-accentués du nom d'un fichier fourni en paramètre.
         * Remplace les points du nom du fichier par un espace (sauf le dernier point)
         * @param {File} file fichier dont on va évaluer le nom (propriété .name)
         * @return {File} retourne un nouveau fichier avec un nom valide
         * ou bien retourne le fichier d'origine si celui-ci ne comportait pas de caractères à modifier
         */
        renameFileWithoutSpecialChar: (file) => {
          let regex = /[éèàâêëîçïôöùûü]/g;
          // compte le nombre de points dans la chaîne de caractères
          let count = ((file.name).match(/\./g) || []).length;

          if (regex.test(file.name) || count > 1) {

            const blob = file.slice(0, file.size, file.type);
            let validFileName = file.name;

            if (regex.test(file.name)) {
              validFileName = (file.name).normalize('NFD').replace(/[\u0300-\u036f]/g, '');
            }
            if (count > 1) {
              // Si la chaîne de caractères contient plusieurs points
              // Remplace tous les points sauf le dernier par un espace
              validFileName = validFileName.replace(/\.(?=[^.]*\.)/g, ' ');
            }
            return new File([blob], validFileName, {type: file.type});
          }
          return file;
        },

        /**
         * Convertit une couleur de hexa en rgb.
         * Reprise d'une méthode un peu dégueu de maFeature.js
         * @param {string} hex code hexadécimal de la couleur
         * @param {number|null} alpha valeur de transparence
         * @param {number|undefined|null} pDarker
         * @return {string} couleur en rgb ou rgba
         */
        hexToRGB: (hex, alpha, pDarker = null) => {
          let darker = pDarker !== undefined && pDarker !== null ? pDarker : 0;
          let r = parseInt(hex.slice(1, 3), 16),
            g = parseInt(hex.slice(3, 5), 16),
            b = parseInt(hex.slice(5, 7), 16);

          r -= darker;
          if (r < 0) {
            r = 0;
          }
          g -= darker;
          if (g < 0) {
            g = 0;
          }
          b -= darker;
          if (b < 0) {
            b = 0;
          }
          if (alpha) {
            return 'rgba(' + r + ', ' + g + ', ' + b + ', ' + alpha + ')';
          }
          else {
            return 'rgb(' + r + ', ' + g + ', ' + b + ')';
          }
        },

        /**
         * Convertit une couleur rgb en hexadécimal. Ne tient pas compte de la transparence.
         * Méthode récupérée de la directive esrisymbolstyle.js
         * @param r couleur rouge
         * @param g couleur vert
         * @param b couleur bleu
         * @return {string} couleur en hexadécimal (ex. rgb(0,0,0) => #000000);
         */
        rgbToHex: (r, g, b) => {
          const componentToHex = (c) => {
            const hex = c.toString(16);
            return hex.length === 1 ? '0' + hex : hex;
          };
          return (
            '#' + componentToHex(r) + componentToHex(g) + componentToHex(b)
          );
        },

        /**
         * Remplace les symboles générés par une mauvais encodage de texte
         * lors de la traduction de caratères accentués
         * Par exemple lorsque le caractère accentué "é" est encodé en "Ã©"
         * @param {string} badEncodedString chaîne de caractères mal encodée
         * lors de la traduction d'un texte en geojson
         * @return {string} chaine de caractère où les symboles sont remplacés
         * par les caractère sans accent correspondant (ex. "Ã©" est remplacé par "e")
         */
        restaureAccentCharacters: (badEncodedString) => {
          return badEncodedString.replace(/Ã©/g, 'e').replace(/Ã¨/g, 'e').replace(/Ã´/g, 'o')
            .replace(/Ãa/g, 'a').replace(/Ã¢/g, 'a');
        },

        /**
         * Transforme une valeur de taille css en numérique. Ex '16px' => 16
         * @param sizeValueAsString taille en string dont on veut extraire
         * la valeur numérique sans le préfixe
         * @return {number} valeur numérique de la taille fournie en paramètre
         */
        isolateCssSizeNumericValue: (sizeValueAsString) => {
          if (sizeValueAsString) {
            let numericSizeValue = '';
            const isNumber = (char) => {
              return /^\d+$/.test(char);
            };
            for (let i = 0; i < sizeValueAsString.length; i++) {
              if (isNumber(sizeValueAsString[i]) || (i > 0 && sizeValueAsString[i] === '.')) {
                numericSizeValue += sizeValueAsString[i];
              }
            }
            if (numericSizeValue.length > 0) {
              return Number.parseInt(numericSizeValue);
            }
            else {
              console.error('isolateCssSizeNumericValue: impossible d\'extraire la valeur de la taille ', sizeValueAsString);
            }
          }
          return 0;
        },
        /**
         * Evite la saisie de texte dans le cas des navigateurs firefox et safari
         * @param {KeyboardEvent} event à la pression de la touche clavier
         * @param {boolean} isDotAllowed à la pression de la touche clavier
         * @return {boolean} <code>true</code> si le texte a été échappé
         */
        skipTextInNumericInput: (event, isDotAllowed = true) => {
          const browsersWithFaillingInputNum = ['firefox', 'safari'];
          if (browsersWithFaillingInputNum.includes(checkBrowser())) {
            return preventTextTyping(event, isDotAllowed);
          }
          return false;
        },
        /**
         * Rogne une image HTML à partir de l'angle HG
         * @param {HTMLImageElement} imageElement source de l'image originale
         * @param {number} width largeur de l'image en sortie
         * @param {number} height hauteur de l'image en sortie
         * @return {HTMLImageElement} image rognée
         */
        resizeImage: (imageElement, width, height) => {
          if(imageElement !== null && Number.isInteger(width) && Number.isInteger(height)) {
            // Créer un canvas
            const canvas = document.createElement('canvas');
            const ctx = canvas.getContext('2d');

            // Définir les dimensions du canvas pour la nouvelle image (20x20)
            canvas.width = width;
            canvas.height = height;

            // Dessiner l'image originale sur le canvas en la rognant
            ctx.drawImage(imageElement, 0, 0, 20, 20, 0, 0, 20, 20);

            // Créer une nouvelle image avec le résultat rogné
            const croppedImage = new Image();
            croppedImage.src = canvas.toDataURL(); // Convertir le canvas en une image

            // Utiliser croppedImage comme nécessaire (l'ajouter à la page, etc.)
            return croppedImage;
          }
          return null;
        },
        hasVerticalScrollbar: (element) => {
          return element.scrollHeight > element.clientHeight;
        },
        deepCopy : (srcOcject) => {
          const jsonString = JSON.stringify(srcOcject);
          return JSON.parse(jsonString);
        },
        /**
         * Affiche une boîte de dialogue d'erreur stylisée avec SweetAlert
         * Le but est de simplifier l'appel et de gérer des barres
         * de défilement pour le cas où le message serait trés long.
         * @param {string} message - Le message d'erreur à afficher
         * @example
         * // Affiche une boîte de dialogue d'erreur
         * jsUtils.swalHtmlError('Une erreur est survenue');
         */
        swalHtmlError: (message) => {
          swal({
            title: $filter('translate')('common.error'),
            text: swalHtmlErrorMessage(message, $filter),
            html: true,
            type: 'error',
            confirmButtonColor: '#428bca',
            showCancelButton: false,
            confirmButtonText: $filter('translate')('common.ok'),
            showConfirmButton: true
          });
        },

        /**
         * Mapping des extensions de fichiers vers leurs types MIME correspondants.
         * Utilisé pour définir le Content-Type lors de la création de Blobs ou
         * pour les téléchargements.
         *
         * @property {Object} applicationTypes - Dictionnaire des types MIME par extension
         * @property {Object} applicationTypes.pdf - Type MIME pour les fichiers PDF
         * @property {Object} applicationTypes.xls - Type MIME pour les fichiers Excel (.xls)
         * @property {Object} applicationTypes.rtf - Type MIME pour les fichiers Rich Text Format
         * @property {Object} applicationTypes.xlsx - Type MIME pour fichiers Excel modernes (.xlsx)
         * @property {Object} applicationTypes.doc - Type MIME pour les fichiers Word (.doc)
         * @property {Object} applicationTypes.docx - Type MIME pour fichiers Word modernes (.docx)
         * @property {Object} applicationTypes.odt - Type MIME pour les fichiers OpenDocument Text
         * @property {Object} applicationTypes.ods - Type MIME pour OpenDocument Spreadsheet
         */
        applicationTypes : {
          pdf: {type: 'application/pdf'},
          xls: {type: 'application/vnd.ms-excel'},
          rtf: {type: 'application/rtf'},
          xlsx: {type: 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet'},
          doc: {type: 'application/msword'},
          docx: {type: 'application/vnd.openxmlformats-officedocument.wordprocessingml.document'},
          odt: {type: 'application/vnd.oasis.opendocument.text'},
          ods: {type: 'application/vnd.oasis.opendocument.spreadsheet'}
        }
      };
    };

    this.$get.$inject = [
      '$filter',
      '$location',
      '$window',
      '$http',
      '$q',
      '$timeout',
      'FunctionFactory',
      'ConfigFactory',
      'FeatureTypeFactory',
      '$rootScope'
    ];
  });

  return mod;
});
