'use strict';
define(function() {
  var gcelement = function (EditFactory, EditTypesFactory, $q, EditRulesFactory, gclayers,
    gcRestrictionProvider, CopyPasteAttributeFactory, bizeditProvider,
    FeatureTypeFactory, $timeout, FeatureAttachmentFactory, gaJsUtils, mapJsUtils,
    BizEditFactory, $filter, EditDescription) {
    /*
    Gestion de la validation des éditions métier.
    Ci-suit le graphe des appels.

    [Clic utilisateur]
      ↓
    scope.save()
      ├─ Si geometryStatus === 'toReverse' → reverseLine()
      └─→ performSaveAndPostSaving()
              ├─→ performSaving()
              ├─→ performPostSaving()
              ├─→ rulesOfRelatedFeatures()
              └─→ processFtiRecords()

    performSaveAndPostSaving()
    ├── performSaving()
    │   ├── prepareFeatures()
    │   │   ├── getOrCreateRecordsObject()
    │   │   └── (traitement des features principales et secondaires)
    │   │
    │   ├── extractDeposesFromAllRecords()
    │   │   └── cleanRecords()
    │   │
    │   ├── deposeCheckConsistencies()
    │   │   └── deposeConsistencyErrors()
    │   │
    │   └── processFtiRecords()
    │       └── runOperation()
    │           ├── hasOperation()
    │           ├── operationFromList()
    │           ├── getCallbackSuccess()
    │           │   └── setSaveValidatedMarker()
    │           └── getCallbackError()
    │
    └── performPostSaving()
        └── rulesOfRelatedFeatures()
            ├── EditRulesFactory.executePostSavingRules()
            └── performSaving()  // Appel récursif pour les mises à jour post-sauvegarde

    // Autres fonctions importantes
    copyDeposeFeatureAttachment()
    reverseLine()
      └── reverseRelatedFeaturesGeometries()
    */
    return {
      templateUrl: 'js/XG/widgets/mapapp/bizedition/views/bizeditsave.html',
      restrict: 'E',
      scope: false,
      link: function(scope) {
        var format = new ol.format.GeoJSON();
        var templatefeatcollect = {
          type: 'FeatureCollection',
          features: [],
        };
        /**
         * Rempli le tableau features des objets graphiques encodés en json,
         * destinés à la sauvegarde.
         * @returns {undefined}
         */
        function prepareFeatures(editdescription) {
          /**
           * Récupére ou crée si il n'existe pas, l'objet associé à une couche
           * et contenant les features à ajouter, mettre à jour, supprimer.
           * adds et updates contienent une collection de Feature en Json.
           * deletes ne contient que les id. f_XX contient les ol.Features.
           *
           * @param {type}
           *            ftiUid
           * @returns
           */
          function getOrCreateRecordsObject(ftiUid, historicFtiUid) {
            if (allRecordsToDo[ftiUid] == undefined) {
              allRecordsToDo[ftiUid] = {
                adds: angular.copy(templatefeatcollect),
                updates: angular.copy(templatefeatcollect),
                deletes: [],
                toHistorize: angular.copy(templatefeatcollect),
                toDepose: angular.copy(templatefeatcollect),
                historicFtiUid: historicFtiUid,
                f_adds: [],
                f_updates: [],
                f_deletes: [],
                f_toHistorize: [],
                f_toDepose: [],
              };
            }
            return allRecordsToDo[ftiUid];
          }

          // Préparation des données pour la sauvegarde de la session
          // d'édition (scope.editdescription)
          var allRecordsToDo = {};

          // Traitement de l'objet principal //
          if (editdescription.editedfeature.saved != true) {
            var recordsObject = getOrCreateRecordsObject(
              editdescription.fti.uid,
              editdescription.fti.historicFtiUid
            );

            var jsoneditedfeature = format.writeFeatureObject(
              editdescription.editedfeature
            );
            if (jsoneditedfeature.properties == null) {
              jsoneditedfeature.properties = {};
            }
            jsoneditedfeature['id'] = jsoneditedfeature['id']
                || editdescription.editedfeature.id_;
            jsoneditedfeature['fti'] = editdescription.fti.uid;
            delete jsoneditedfeature.properties.objectIsCopied;
            if (
              editdescription.editType ===
              EditTypesFactory.editTypes.add.name
            ) {
              recordsObject.adds.features.push(jsoneditedfeature);
              recordsObject.f_adds.push(editdescription.editedfeature);
            } else if (editdescription.editType === EditTypesFactory.editTypes.update.name
                || editdescription.editType === EditTypesFactory.editTypes.updateattributes.name
                || editdescription.editType === EditTypesFactory.editTypes.reverse.name) {
              recordsObject.updates.features.push(jsoneditedfeature);
              recordsObject.f_updates.push(editdescription.editedfeature);
            }
            // Pour la suppression
            else if (
              editdescription.editType ===
              EditTypesFactory.editTypes.delete.name
            ) {
              recordsObject.deletes.push(jsoneditedfeature.id);
              recordsObject.f_deletes.push(editdescription.editedfeature);
            }

            // Historisation eventuelle
            else if (
              editdescription.editType ===
              EditTypesFactory.editTypes.tohistorize.name
            ) {
              recordsObject.toHistorize.features.push(jsoneditedfeature);
              recordsObject.f_toHistorize.push(
                editdescription.editedfeature
              );
            } else if (
              editdescription.editType ===
              EditTypesFactory.editTypes.todepose.name
            ) {
              recordsObject.toDepose.features.push(jsoneditedfeature);
              recordsObject.f_toDepose.push(
                editdescription.editedfeature
              );
            }
          }
          // Traitement des objets secondaires //
          for (
            var i = 0;
            i < editdescription.relatedfeatures.length;
            i++
          ) {
            var related = editdescription.relatedfeatures[i];
            var feature = related.feature;
            // On ne sauve pas le feature si celui-ci est marqué comme étant
            // dejà sauvegardé.
            if (feature.saved === true) continue;

            var ftuid = related.fti.uid;
            jsoneditedfeature = format.writeFeatureObject(feature);
            jsoneditedfeature['fti'] = ftuid;
            if (jsoneditedfeature.properties == null) {
              jsoneditedfeature.properties = {};
            }
            let recordsObject = getOrCreateRecordsObject(
              ftuid,
              related.historicFtiUid
            );

            delete jsoneditedfeature.properties.objectIsCopied;
            if (related.editType === EditTypesFactory.editTypes.add.name) {
              recordsObject.adds.features.push(jsoneditedfeature);
              recordsObject.f_adds.push(feature);
            } else if (
              related.editType === EditTypesFactory.editTypes.update.name ||
              related.editType === EditTypesFactory.editTypes.updateattributes.name ||
              related.editType === EditTypesFactory.editTypes.reverse.name
            ) {
              recordsObject.updates.features.push(jsoneditedfeature);
              recordsObject.f_updates.push(feature);
            }
            // Pour la suppression, voir comment récupérer l'identifiant du
            // feature
            else if (
              related.editType === EditTypesFactory.editTypes.delete.name
            ) {
              recordsObject.deletes.push(jsoneditedfeature.id);
              recordsObject.f_deletes.push(feature);
            }
            // Historisation eventuelle
            else if (
              related.editType === EditTypesFactory.editTypes.tohistorize.name
            ) {
              recordsObject.toHistorize.features.push(jsoneditedfeature);
              recordsObject.f_toHistorize.push(feature);
            } else if (
              related.editType === EditTypesFactory.editTypes.todepose.name
            ) {
              recordsObject.toDepose.features.push(jsoneditedfeature);
              recordsObject.f_toDepose.push(feature);
            }
          }

          return allRecordsToDo;
        }


        const hasPostSavingRules = (fti) => {
          return fti.rules.some(rule => rule.type === 'PostSave');
        };


        /**
         * Formate un rapport en HTML avec mise en forme des erreurs
         * @param {string} report - Le rapport texte à formater
         * @returns {string} Le rapport formaté en HTML avec mise en forme
         */
        const messageFromReport = (report) => {
          // Styles CSS pour le conteneur principal
          const stylep = 'margin-left: 10px; text-align: left;';
          // Styles pour le titre du rapport
          const styleq = 'margin-left: 2px; text-align: left; '
            + 'font-weight: bold; font-size: 14px;';
          // Style pour les lignes d'erreur
          const errorStyle = 'color: red; font-weight: bold;';

          // Traitement du rapport : coloration des lignes d'erreur et gestion des sauts de ligne
          const formattedReport = report
            .split('\n')
            .map(line => line.startsWith('[ERREUR]')
              ? `<span style="${errorStyle}">${line}</span>`
              : line)
            .join('<br>');

          // Construction du HTML final avec le titre traduit
          const title = $filter('translate')('common.report') + ':';
          return '<div style="' + styleq + '">' + title + '</div>'
            + '<div style="' + stylep + ' max-height: 300px; overflow-y: auto; '
            + 'padding: 5px; border: 1px solid #ddd; border-radius: 4px;">'
            + formattedReport + '</div>';
        };


        /**
         * Gère l'affichage du rapport retourné (y compris les erreurs)
         * dans une boîte de dialogue SweetAlert.
         * Le rapport n'est affiché qu'en cas d'erreur.
         * @param {Object} data - Les données de la réponse contenant les informations d'erreur
         * @param {Object} data.info - Contient les détails des erreurs
         * @param {number} data.info.errorCount - Nombre d'erreurs détectées
         * @param {string} data.info.report - Rapport d'erreurs à afficher
         * @returns {boolean} true si des erreurs ont été traitées, false sinon
         */
        const manageErrors = (data) => {
          if (data.info && data.info.errorCount !== 0) {
            swal({
              title: $filter('translate')('bizedition.deposeErrorTitle'),
              text: messageFromReport(data.info.report),
              html: true,
              type: 'error',
              confirmButtonColor: '#428bca',
              showCancelButton: false,
              confirmButtonText: $filter('translate')('common.ok'),
              showConfirmButton: true
            });
            return true;
          }
          return false;
        };


        /**
         * [save description]
         *
         * @return {Promise} [description]
         */
        function performSaving(editdescription) {
          const defer = $q.defer();
          const srid = scope.map.getView().getProjection().getCode();
          const promises = [];

          if (!editdescription) {
            editdescription = scope.editdescription;
          }



          const hasOperation = (records, operation) => {
            if(operation === 'deletes') {
              return records
                && records[operation]
                && records[operation].length > 0;
            }
            return records
              && records[operation]
              && records[operation].features
              && records[operation].features.length > 0;
          };


          const operationFromList = (operation) => {
            const operations = {
              adds: 'add',
              updates: 'update',
              toHistorize: 'historicize',
              toDepose: 'depose',
              deletes: 'remove'
            };
            return operations[operation];
          };


          /**
           * Exécute une opération d'édition (ajout, mise à jour, suppression, etc.)
           * sur un ensemble d'enregistrements
           * @param {Object} records - Objet contenant les enregistrements à traiter
           * @param {string} operation - Type d'opération à effectuer
           * (adds, updates, deletes, toHistorize, toDepose)
           * @param {string} ftiUid - Identifiant unique du type de feature
           * @param {string} srid - Système de référence des coordonnées
           * @returns {Promise|undefined} Une promesse résolue lorsque l'opération
           * est terminée, ou undefined si l'opération n'est pas applicable
           */
          const runOperation = (records, operation, ftiUid, srid, promises) => {
            if (hasOperation(records, operation)) {
              let callBackSuccess = getCallbackSuccess(records, ftiUid, operation);
              let callBackError = getCallbackError(records, ftiUid, operation);

              // -- "updates" -> "update"
              const action = operationFromList(operation);
              const param3 = action === 'remove' ? undefined : srid;
              promises.push(EditFactory[action](ftiUid, records[operation], param3)
                .then(callBackSuccess, callBackError));
            }
          };


          scope.allRecords = prepareFeatures(editdescription);
          const processFtiRecords = (keys, index, defer, records) => {
            if(index >= keys.length) {
              defer.resolve();
              return;
            }
            if(!defer) {
              defer = $q.defer();
            }
            const ftiUid = keys[index];
            const recordsObject = records[ftiUid];

            const promises = [];
            runOperation(recordsObject, 'adds', ftiUid, srid, promises);
            runOperation(recordsObject, 'updates', ftiUid, srid, promises);
            runOperation(recordsObject, 'deletes', ftiUid, srid, promises);
            runOperation(recordsObject, 'toHistorize', ftiUid, srid, promises);
            runOperation(recordsObject, 'toDepose', ftiUid, srid, promises);
            if(promises.length !== 0) {
              $q.all(promises).then(() => {
                processFtiRecords(keys, index + 1, defer, records);
              });
            }
            return defer.promise;
          };


          /**
           * KIS-3768
           * Nettoie les enregistrements d'édition pour ne pas envoyer
           * les actions de dépose dans le apply edits de ArcGIS..
           *
           * @param {string} key - La clé identifiant l'ensemble d'enregistrements à nettoyer
           *                       Il s'agit de l'UID du FTI des objets à traiter.
           *
           * Cette fonction effectue deux actions principales :
           * 1. Supprime les entrées de `allRecords` qui n'ont aucune
           *    opération d'ajout, suppression ou mise à jour et
           *    qui n'ont donc que des opérations de dépose.
           * 2. Pour les entrées restantes, réinitialise le tableau `toDepose` à vide
           *    Les déposes ayant au préalable été extraites pour traitement hors apply edits.
           */
          const cleanRecords = (key) => {
            if((!scope.allRecords[key].adds
              || !scope.allRecords[key].adds.features
              || scope.allRecords[key].adds.features.length===0)
              && (!scope.allRecords[key].deletes
                || !scope.allRecords[key].deletes.features
                || scope.allRecords[key].deletes.features.length===0)
              && (!scope.allRecords[key].updates
                || !scope.allRecords[key].updates.features
                || scope.allRecords[key].updates.features.length===0)
            ) {
              // -- Aucune opération à part la dépose, on n'a donc rien à envoyer au "applyEdits"
              delete scope.allRecords[key];
            }
            else {
              // -- Il y a des opérations autre que la dépose,
              // -- on va donc envoyer au "applyEdits", mais, sans la dépose
              scope.allRecords[key].toDepose = [];
            }
          };


          /**
           * KIS-3768
           * Enlever les déposes du allRecords pour le cas du applyEdits de ArcGIS.
           * @returns {Object} Les déposes à envoyer au applyEdits
           *                   ou NULL s'il n'y a pas de déposes.
           */
          const extractDeposesFromAllRecords = () => {
            const deposes = {};
            let hasDeposes = false;
            for (const key in scope.allRecords) {
              if (scope.allRecords[key].toDepose
                && scope.allRecords[key].toDepose.features
                && scope.allRecords[key].toDepose.features.length > 0) {
                deposes[key] = {
                  toDepose: scope.allRecords[key].toDepose,
                  f_toDepose: scope.allRecords[key].f_toDepose
                };
                cleanRecords(key);
                hasDeposes = true;
              }
            }
            return hasDeposes ? deposes : null;
          };


          /**
           * KIS-3768
           * Formate les messages d'erreur de cohérence pour l'affichage à l'utilisateur.
           *
           * @param {string} ftiUid - L'identifiant unique du type d'entité (Feature Type)
           * @param {Object} errors - Objet contenant les différentes catégories d'erreurs
           * @param {Array<Object>} [errors.consistencyErrors] - Erreurs de cohérence métier
           * @param {string} errors.consistencyErrors[].message - Message d'erreur de cohérence
           * @param {Array<string>} [errors.configErrors] - Erreurs de configuration
           * @param {Array<string>} [errors.serverErrors] - Erreurs serveur
           * @returns {string} - Message HTML formaté pour l'affichage
           *
           * La fonction génère une structure HTML hiérarchique avec un style cohérent :
           * - Titre principal : Nom du type d'entité
           * - Section "Cohérence" : Liste des erreurs de cohérence
           * - Section "Configuration" : Liste des erreurs de configuration
           * - Section "Serveur" : Liste des erreurs serveur
           *
           * Chaque section n'est affichée que si elle contient des erreurs.
           * Le style est appliqué en ligne pour une meilleure portabilité.
           */
          const deposeConsistencyErrors = (ftiUid, errors) => {
            // -- Styles CSS en ligne pour le formatage du message d'erreur
            const styleh2 = 'font-size: 16px; font-weight: bold; text-align: left;';
            const styleh3
              = 'font-size: 14px; font-weight: bold; text-align: left; margin-left: 15px;';
            const stylep = 'margin-left: 30px; text-align: left;';

            // -- Récupération du nom du type d'entité à partir de son UID
            const ftiName = FeatureTypeFactory.getFeatureTypeNameByUid(ftiUid);
            // -- Initialisation du message avec le nom du type d'entité
            let message = '';

            // -- Ajout des erreurs de cohérence si elles existent
            if (errors.consistencyErrors && errors.consistencyErrors.length != 0) {
              message += '<div style="' + styleh3 + '">  Cohérence:</div><br>';
              message += '<div style="' + stylep + '">'
                + errors.consistencyErrors.map(err => err.message).join('<br>');
              message += '</div><br>';
            }
            // -- Ajout des erreurs de configuration si elles existent
            if (errors.configErrors && errors.configErrors.length != 0) {
              message += '<div style="' + styleh3 + '">  Configuration:</div><br>';
              message += '<div style="' + stylep + '">' + errors.configErrors.join('<br>');
              message += '</div><br>';
            }
            // -- Ajout des erreurs serveur si elles existent
            if (errors.serverErrors && errors.serverErrors.length != 0) {
              message += '<div style="' + styleh3 + '">  Serveur:</div><br>';
              message += '<div style="' + stylep + '">' + errors.serverErrors.join('<br>');
              message += '</div><br>';
            }
            if (message.length!==0) {
              message = '<div style="' + styleh2 + '">' + ftiName + '</div><br>' + message;
            }
            return message;
          };


          /**
           * Modification du KIS-3370: vérifie la bonne cohérence des champs
           * de TOUTES LES DEPOSES avant dépose.
           * Si des problèmes sont constatés, ils sont listés dans un SWeet ALert.

           * @param {*} ftiUidList
           * @returns
           */
          const deposeCheckConsistencies = (ftiUidList) => {
            const defer = $q.defer();
            BizEditFactory.checkDeposeConsistency(ftiUidList).then(
              res => {
                if (res.data) {
                  let errorMsg = '';
                  for (const ftiUid of ftiUidList) {
                    if (res.data[ftiUid]) {
                      if (errorMsg!=='') {
                        errorMsg += '<br>';
                      }
                      errorMsg += deposeConsistencyErrors(ftiUid,res.data[ftiUid]);
                    }
                  }
                  if (errorMsg) {
                    const errorContainerStyle = 'max-height: 350px; overflow-y: auto; '
                      + 'padding: 10px; border: 1px solid #ddd; border-radius: 4px;';
                    const errorContent = `<div style="${errorContainerStyle}">${errorMsg}</div>`;

                    swal({
                      title: $filter('translate')('bizedition.deposeErrorTitle'),
                      text: errorContent,
                      html: true,
                      type: 'error',
                      confirmButtonColor: '#428bca',
                      showCancelButton: false,
                      confirmButtonText: $filter('translate')('common.ok'),
                      showConfirmButton: true
                    });
                    defer.reject();
                  }
                  else {
                    defer.resolve();
                  }
                }
              }
            );
            return defer.promise;
          };


          const areMissingAttributes
            = Object.keys(scope.allRecords || {}).reduce((missingAttributes, featureUid) => {
              const fti = FeatureTypeFactory.resources.featuretypes
                .find(featuryType => featuryType.uid === featureUid) || { attributes: [] };
              scope.allRecords[featureUid].adds.features.forEach((feature) => {
                const missingMandatoryAttributes = bizeditProvider.getMissingMandatoryAttributes(
                  fti.attributes, 1, undefined, feature.properties
                );
                if (missingMandatoryAttributes.length) {
                  missingAttributes = true;
                }
              });
              return missingAttributes;
            }, false);

          if (areMissingAttributes) {
            require('toastr').error(
              'Des attributs obligatoires sont à renseigner dans les objets dépendants');
            scope.updatesToSave = true;
            defer.reject();
            return defer.promise;
          }

          // -- Traitement des déposes
          // -- -1- Extraire les déposes de la liste des éditions
          const deposes = extractDeposesFromAllRecords(scope);
          const keys = [];
          for (const key in deposes) {
            keys.push(key);
          }
          // -- -2- S'il y a des déposes, vérifier qu'il n'y aura pas de problème à les gérer
          const deferDeposes = $q.defer();
          if(deposes) {
            deposeCheckConsistencies(keys, deposes).then(() => {
              deferDeposes.resolve();
            });
          }
          else {
            deferDeposes.resolve();
          }

          deferDeposes.promise.then(() => {
            // -- Traitement des éditions
            // -- (pas de problèmes avec les déposes, ou pas de déposes dans les éditions).
            if (scope.selectfti.type === 'esri') {
              const argisDefer = $q.defer();
              if(deposes) {
                promises.push(argisDefer.promise);
                processFtiRecords(keys, 0, argisDefer, deposes);
              }
              promises.push(EditRulesFactory.applyEdits(scope));
              $q.all(promises).then(() => {
                defer.resolve();
              });
            } else {
              // -- Remise en place des dépose cas non ArcGIS
              for (const key in deposes) {
                scope.allRecords[key] = deposes[key];
              }
              // -- Pour chaque couche, enregistrement des features ajoutés,
              // -- mis à jour et supprimés
              for (const key in scope.allRecords) {
                if (keys.indexOf(key)===-1) {
                  keys.push(key);
                }
              }
              return processFtiRecords(keys, 0, defer, scope.allRecords);
            }
          });


          function getCallbackSuccess(recordsObject, ftiUid, recordsType, defer) {
            return function(result) {
              const data = result.data;

              if (manageErrors(data)) {
                return;
              }

              // Si la reponse n'est pas un objet du service d'édition, il y a
              // une erreur
              if (data.create == undefined) {
                const msg = 'Erreur serveur: ' + data.message + ', details:' + data.details[0];
                require('toastr').error(msg);
                console.error(msg);
                return;
              }

              if (recordsType === 'adds') {
                if (data.create.length === recordsObject.adds.features.length) {
                  // Marquage de chaque ol.Feature comme enregistré avec succès.
                  setSaveValidatedMarker(recordsObject.f_adds);
                  // Récupération des features sauvées avec un id affecté,
                  // encodés en json.
                  // Si les features ne contiennent pas les id, c dernier se
                  // trouve quand meme dans data.create[i]
                  for (let i = 0; i < data.create.length; i++) {
                    let jsonFeature;
                    try {
                      jsonFeature = JSON.parse(data.create[i].json);
                      // KIS-3631 Ajouté pour éviter les identifiants 'undefined'
                      // dans les enregistrements à ajouter
                      recordsObject.f_adds[i].setId(jsonFeature.id);
                      recordsObject.adds.features[i].id = jsonFeature.id;

                      // KIS-3298: on doit créer les fieldData des objets enregistrés
                      // sans ouvrir la popup d'édition attributaire
                      // rappel: les fieldData stockent le nom du sous-dossier UPLOAD
                      // dans lequel on été chargé les attachments
                      const fti
                        = bizeditProvider.getFtiFromJsonFeature(jsonFeature, data.create, 0);
                      if (fti && !Object.prototype.hasOwnProperty.call(scope.fieldDatas, fti.uid)) {

                        // rang de l'objet parmi tous les objets nouvellement créés
                        // du composant (-1 si objet principal)
                        const indexFeature = editdescription.relatedfeatures
                          .findIndex(relFeat => relFeat.id === jsonFeature.id);
                        scope.createAttributePopupFieldDatas(fti, indexFeature, null);
                      }

                    } catch (e) {
                      console.error(
                        'Error de parsing de result.data.create:' +
                          data.create[i],
                        e
                      );
                    }
                  }
                  // On copie tous les fichiers d'un objet avant de passer à l'objet suivant
                  bizeditProvider.copyFilesFromUploadToAttachmentFolder(
                    data.create, 0, scope.fieldDatas);

                  // KIS-3135:  l’enregistrement des objets associés se fait
                  // après enregistrement de l’objet édité
                  // et une fois que son identifiant est récupéré
                  // (enregistrement en postSave si présence de règle postSave)
                  //KIS-3631: Pour éviter la perte de l'objet editdescription
                  if (data.create.length === 1 && !hasPostSavingRules(editdescription.fti)) {
                    EditRulesFactory.bizEditSaveAssociatedFeatures(data.create, editdescription);
                  }
                }
              } else if (recordsType === 'updates') {
                if (data.update.length === recordsObject.updates.features.length) {
                  setSaveValidatedMarker(recordsObject.f_updates);
                }
              } else if (recordsType === 'deletes') {
                if (data.delete.length === recordsObject.deletes.length) {
                  setSaveValidatedMarker(recordsObject.f_deletes);
                }
              } else if (recordsType === 'toHistorize') {
                if (Object.prototype.hasOwnProperty.call(data, 'historic')
                  && Array.isArray(data.historic)
                  && data.historic.length === recordsObject.toHistorize.features.length) {
                  setSaveValidatedMarker(recordsObject.f_toHistorize);
                }
              } else if (recordsType === 'toDepose') {
                // Comparer les id des features pour vérifier plus précisement
                // lesquels ont bien été sauvés.
                if (data.depose.length === recordsObject.toDepose.features.length) {
                  copyDeposeFeatureAttachment(data.depose, recordsObject.toDepose.features, ftiUid);
                  setSaveValidatedMarker(recordsObject.f_toDepose);
                }
              }
              gclayers.refreshlayerByid(ftiUid, scope.map);
              if (data.errors.length > 0) {
                console.error(
                  'Sauvegarde de type ' +
                    recordsType +
                    ' pour la couche uid:' +
                    ftiUid +
                    ' non effectuée !\n Cause:' +
                    data.errors[0]
                );
              }
              // enlève les fonctions spécifiques du menu contextuel. Préserve zoom +/-
              scope.resetContextMenu();
              if (defer) {
                defer.resolve();
              }
            };
          }

          function setSaveValidatedMarker(features) {
            for (var i = 0; i < features.length; i++) {
              features[i].saved = true;
            }
          }

          function getCallbackError(recordsObject, ftiUid, recordsType, defer) {
            return function(result) {
              if (editdescription.editType === EditTypesFactory.editTypes.reverse.name) {
                cancelReversedGeometries();
              }
              gcRestrictionProvider.showDetailsErrorMessage(result);
              console.error('Erreur de sauvegarde de type ' + recordsType
                + ' pour la couche uid:' + ftiUid + ' ,response:' + result);
              if (defer) {
                defer.reject();
              }
            };
          }

          // Renvoi une promesse qui sera résolue lorsque toutes les promesses
          // du tableau seront résolues;
          return defer.promise;
        } // end performSaving


        /**
         * Reconstruire la couche openLayers
         * contenant les objets d'accroche (SNAP).
         */
        scope.initializeSnapLayer = (descriptionObject) => {
          $timeout(() => {
            scope.reset();
            scope.saveinProgress = false;
            // l'exécution de cette méthode depuis bizeditwidget rend editdescription undefined
            // dans le cas de l'edition metier (maj) celle là casse le snap
            // const editdescription
            //   = scope.editdescription ? scope.editdescription : descriptionObject;
            // EditRulesFactory.executeInitRules(editdescription,
            //   scope.selectfti,scope.map);
          });
        };


        /**
         * Exécute les règles post-sauvegarde pour les objets liés et gère leur mise à jour
         *
         * Cette fonction :
         * - Filtre les objets liés qui ne sont pas en mode suppression
         * - Crée un "editdescription" avec comme objet principal
         *   l'objet lié avec les informations nécessaires
         * - Exécute ainsi les règles post-sauvegarde pour chaque objet lié
         * - Met à jour les objets liés dans la session d'édition
         * - Gère la copie d'attributs si nécessaire
         * - Marque les objets comme sauvegardés
         * - Déclenche une sauvegarde supplémentaire si nécessaire
         *
         * @param {Object} defer - Objet promesse pour résoudre l'opération
         * @param {Object} res - Résultat de la sauvegarde précédente
         * @returns {void}
         */
        const rulesOfRelatedFeatures = (defer, res) => {
          const promises = [];
          if (scope.editdescription && Array.isArray(scope.editdescription.relatedfeatures)) {
            const features = scope.editdescription.relatedfeatures.filter((relatedFeat) => {
              return EditTypesFactory.editTypeIsNotDelete(relatedFeat.editType);
            });
            if (features.length > 0) {
              let i=0;
              for (let relatedFeat of features) {
              // Reconstruit les related features pour chaque related feature
                let newRelatedFeat = EditDescription.getTemplateRelatedFeature(
                  relatedFeat,scope.editdescription, true);
                newRelatedFeat.index = i;
                if (hasPostSavingRules(newRelatedFeat.fti)) {
                  const promise = EditRulesFactory.executePostSavingRules(
                    newRelatedFeat,
                    scope.map,
                    scope.allRecords,
                    i++
                  ).then(() => {
                    newRelatedFeat.editType = 'update';
                    newRelatedFeat.editedfeature.saved = false;
                    newRelatedFeat.feature = newRelatedFeat.editedfeature;
                    // Remplace le related feature original par le nouveau
                    // dans la liste des related features
                    scope.editdescription.relatedfeatures[newRelatedFeat.index] = newRelatedFeat;
                    // Pour exécuter la méthode performSaving
                    newRelatedFeat.performUpdateOnPostsaving = true;
                  });
                  promises.push(promise);
                }
              }
            }
          }
          $q.all(promises).then(()=>{
            //set object from response to populate the pop-up
            const rules = scope.editdescription.fti.rules;
            rules.find( (attribute) => {
              if (attribute.name === 'CopyPasteAttribute') {
                const currentObject = res;
                currentObject['objectIsCopied'] = true;
                CopyPasteAttributeFactory.StoreObject(
                  currentObject,
                  scope.editdescription.fti
                );
              }
            });

            ///// add editdescription.editedfeature.saved as true in case of saving
            if (scope.editdescription.editType === 'add' ||
                scope.editdescription.editType === 'update' ||
                scope.editdescription.editType === 'updateattributes') {
              scope.editdescription.editedfeature.saved = true;
              scope.editdescription.editedfeature.attrEdited = true;
            }

            const promises = [];
            if (scope.editdescription.performUpdateOnPostsaving === true) {
              scope.editdescription.editType = 'update';
              scope.editdescription.editedfeature.saved = false;
              scope.editdescription.relatedFeature = [];
              promises.push(performSaving());
            }
            for (const feature of scope.editdescription.relatedfeatures) {
              if (feature.performUpdateOnPostsaving === true) {
                feature.editType = 'update';
                feature.editedfeature.saved = false;
                feature.relatedfeatures = [];
                promises.push(performSaving(feature));
              }
            }
            $q.all(promises).then(() => { defer.resolve(res); });
          });
        };


        const performPostSaving = () => {
          const defer = $q.defer();
          scope.saveinProgress = false;
          //Une fois les sauvegardes terminées, lancement des règles post-sauvegarde.
          const promise = EditRulesFactory.executePostSavingRules(
            scope.editdescription,
            scope.map,
            scope.allRecords,
            0
          ).then(result => {
            console.log('Traitement terminé avec succès:', result);
          })
            .catch(error => {
              console.error('Erreur lors du traitement:', error);
            });

          promise.then((res) => {
            rulesOfRelatedFeatures(defer, res);
          }
          );
          return defer.promise;
        };


        const performSaveAndPostSaving = () => {
          // Bloquer le rafraîchissement de la carten
          // tant que l'on n'a pas traiter tous les obejts et leurs règles
          // La fonction scope.endsave() déblocera le rafraîchissement de la carte
          gclayers.setNoMapRefresh(true);
          performSaving().then(
            (res) => {
              //-- Dans le cas esri, l'applyedits qui est appelé
              //-- s'occupe aussi du postSaving.
              if ((!res || ! res[0] || res[0].type !== 'esri')
                && scope.editdescription
                && hasPostSavingRules(scope.editdescription.fti)) {
                performPostSaving().then(() => {
                  // -- Attendre que le apply angularj soit fini avant de rafraîchir la carte
                  scope.endsave(scope.allRecords);
                });
              }
              else {
                const defer = $q.defer();
                rulesOfRelatedFeatures(defer, res);
                defer.promise.then(() => {
                  //(reprendre les feature add et update et effectuer des updates)
                  //Règles post-sauvegarde terminées.
                  scope.endsave(scope.allRecords);
                });
              }
            },
            (error) => {
              console.error('performSaveAndPostSaving : ', error);
              scope.saveinProgress = false;
              gclayers.setNoMapRefresh(false);
            }
          );
        };


        scope.save = () => {
          scope.saveinProgress = true;

          if (scope.editdescription.editedfeature != undefined) {
            if (scope.editdescription.geometryStatus === 'toReverse') {
              reverseLine();
              performSaveAndPostSaving();
            } else if (
              scope.editdescription.geometryStatus !== 'modifiedAndNotSaved'
            ) {
              //-- Dans le cas de modification d'objet:
              //-- -- Les EndRules ont été exécutées par activation
              //-- -- de la "coche", il ne reste donc plus qu'à exécuter
              //-- -- les règles PostSaving.
              //-- Dans le cas de création d'objet:
              //-- -- geometryStatus contient "undefined", et les end rules
              //-- -- sont exécutées dans les traitements par défaut,
              //-- -- donc à ne pas exécuter à nouveau.
              performSaveAndPostSaving();
            } else {
              //Execution des règles 'onend'.
              var saveEnd = EditRulesFactory.executeEndRules(
                scope.editdescription,
                scope.editdescription.fti,
                scope.map
              );
              saveEnd.then(
                () => {
                  console.info('EditRulesFactory.executeEndRules done.');
                  console.log('scope.editdescription', scope.editdescription);
                  performSaveAndPostSaving();
                },
                (errorReason) => {
                  if (errorReason && typeof errorReason === 'string') {
                    console.error(
                      'EditRulesFactory.executeEndRules, error:' + errorReason
                    );
                  }
                  $timeout(() => {
                    scope.reset();
                    scope.saveinProgress = false;
                  });
                }
              );
            }
          }
        };

        /**
         * Copie les fichiers attachés d'un objet déposé depuis le dossier
         * d'attachment de cet objet vers le dossier d'attachment de l'objet de dépose.<br>
         * Ex. copie les fichiers attachés depuis le dossier
         * "ATTACHMENTS/Regards/Regards.720" vers le dossier
         * "ATTACHMENTS/Regards_depose/Regards_depose.24"<br>
         * Cette méthode suppose que les tableaux fournis en paramètres
         * soient classés dans le même ordre: deposedFeatures[i] est l'objet déposé
         * de toDeposeFeatures[i].
         * @param {object[]} deposedFeatures tableau d'objets de dépose après création
         *   et enregistrement.
         *   Chaque objet présente l'id de l'objet ainsi qu'une propriété json
         *   constituant l'objet en string json
         * @param {object[]} toDeposeFeatures tableau d'objets à déposer.
         *   Chaque objet présente l'id de l'objet ainsi que les propriétés non nulles
         */
        const copyDeposeFeatureAttachment = (deposedFeatures, toDeposeFeatures, ftiUid) => {
          if (Array.isArray(deposedFeatures) && Array.isArray(toDeposeFeatures)) {
            const deposedLength = deposedFeatures.length;
            const toDeposedLength = toDeposeFeatures.length;
            if (deposedLength > 0 && deposedLength === toDeposedLength) {
              scope.saveinProgress = true;
              const promises = [];
              const fti = FeatureTypeFactory.getFeatureByUid(ftiUid);
              const ftiDepose = FeatureTypeFactory.getFeatureByUid(fti.historicFtiUid);
              for (let i = 0; i < deposedLength; i++) {
                const source = bizeditProvider.buildRequestParameter(toDeposeFeatures[i], fti);
                const feature = JSON.parse(deposedFeatures[i].json);
                const destination
                  = bizeditProvider.buildRequestParameter(feature, ftiDepose);
                promises.push(
                  FeatureAttachmentFactory.copyFilesToFeature(source, destination, [], true));
              }
              $q.all(promises).finally(
                () => {
                  scope.saveinProgress = false;
                }
              );
            }
          }
        };

        /**
         * Inverse le sens de digitalisation d’une ligne ou d'une sélection de lignes
         * au clic sur le bouton "Inverser le sens" (flèches inversées)
         */
        const reverseLine = () => {

          // inverse le sens de l'objet principal
          const editedGeometry = scope.editdescription.editedfeature.getGeometry();
          const backupGeometry = editedGeometry.clone();
          const reversedGeometry = mapJsUtils.getReversedLinearGeometry(editedGeometry);
          scope.editdescription.editedfeature.setGeometry(reversedGeometry);

          // inverse le sens des objets liés linéaires
          const relatedFeaturesBackupGeometries
            = reverseRelatedFeaturesGeometries(scope.editdescription);

          // objet de sauvegarde pour restaurer en cas d'échec (cf. getCallbackError)
          scope.reversingLineData = {
            bypassRules: ['MoveExtremityPoint', 'MoveObjectsOnLine'],
            backupGeometry: backupGeometry,
            relFeatsBackupGeoms: relatedFeaturesBackupGeometries
          };
        };

        /**
         * Inverse le sens de la géométrie de chaque objet relié
         * de la session d'édition ("relatedfeatures").
         * Dans la structure actuelle d'une session d'édition,
         * quand l'utilisateur sélectionne plusieurs objets,
         * le premier objet sélectionné est l'objet principal (editdescription.editedfeature)
         * et les autres objets sélectionnés sont des objets reliés
         * (editdescription.relatedfeatures).
         * @param editDescription objet stockant les caractéristiques d'une session d'édition
         * @return {Map<any, ol.geom.LineString|ol.geom.MultiLineString>} map contenant
         * pour chaque objet relié: l'id ol de chaque objet relié et sa géométrie
         * avant inversion de sens
         */
        const reverseRelatedFeaturesGeometries = (editDescription) => {

          // traite le cas où la session d'édition est nulle ou incorrecte
          if (!gaJsUtils.notNullAndDefined(editDescription)
            || !Array.isArray(editDescription.relatedfeatures)) {
            console.error(
              `reverseRelatedFeaturesGeometries - La session d'édition est nulle`,
              `ou ne possède aucun objet lié : editdescription = `, editDescription
            );
            return new Map();
          }

          // conteneur des géométries d'origine
          const backupGeometries = new Map();

          const lineTypes = ['MultiLineString', 'LineString'];

          // traite chaque objet lié
          for (const relatedFeature of editDescription.relatedfeatures) {

            if (!gaJsUtils.notNullAndDefined(relatedFeature.feature)) {
              console.error(`reverseRelatedFeaturesGeometries - L'objet lié ${relatedFeature.id} ne possède pas de feature openlayers. relatedFeature = `, relatedFeature);
              return backupGeometries;
            }
            const olRelatedFeature = relatedFeature.feature;
            const relatedGeometry = olRelatedFeature.getGeometry();

            // vérifie que l'objet soit bien un linéaire
            if (relatedGeometry !== null && lineTypes.includes(relatedGeometry.getType())) {

              // copie la géométrie d'origine
              const relatedOriginalGeometry = relatedGeometry.clone();

              // recherche l'id de l'objet openlayers pour insertion dans le conteneur
              let featId;
              if (gaJsUtils.notNullAndDefined(olRelatedFeature.getId())) {
                featId = olRelatedFeature.getId();
              } else {
                // dans le cas impossible où une feature existante n'aurait pas d'id,
                // alors on identifie la feature à l'aide d'une propriété custom "id"
                featId = gaJsUtils.guid();
                olRelatedFeature.set('id', featId);
              }
              backupGeometries.set(featId, relatedOriginalGeometry);

              // inverse le sens de la géométrie de l'objet openlayers
              const relatedReversedGeometry = mapJsUtils.getReversedLinearGeometry(relatedGeometry);
              olRelatedFeature.setGeometry(relatedReversedGeometry);
            }
          }

          // retourne le conteneur des géométrie initiales pour permettre une restauration
          // si une erreur serveur intervient
          return backupGeometries;
        };

        /**
         * Lorsque une erreur serveur est levée, alors l'inversion de sens est annulée.
         * Cette méthode permet de restaurer la géométrie des objets liés dans le sens d'origine
         * @param backupGeometries map contenant la géométrie initiale des objets reliés
         *                         de la session d'édition
         * @param editDescription objet contenant les informations de la session d'édition
         */
        const restoreRelatedFeaturesGeometries = (backupGeometries, editDescription) => {
          if (backupGeometries && backupGeometries.size > 0
            && Array.isArray(editDescription.relatedfeatures)) {
            for (const relatedFeature of editDescription.relatedfeatures) {
              const olRelatedFeature = relatedFeature.feature;
              let featId;
              if (gaJsUtils.notNullAndDefined(olRelatedFeature.getId())) {
                featId = olRelatedFeature.getId();
              }
              else if (gaJsUtils.notNullAndDefined(olRelatedFeature.get('id'))) {

                // Dans le cas impossible où une feature existante n'aurait pas d'id,
                // on peut toujours identifier la feature à l'aide
                // d'une propriété custom "id" définie lors de l'inversion de sens
                featId = String(olRelatedFeature.get('id')).slice();

                // supprime la propriété custom "id" pour ne pas générer de conflits
                olRelatedFeature.unset('id');
              }
              if (gaJsUtils.notNullAndDefined(featId) && backupGeometries.has(featId)) {
                olRelatedFeature.setGeometry(backupGeometries.get(featId));
              }
            }
          }
        };

        /**
         * Dans le cas d'une inversion de sens d'objets linéaires et
         * dans le cas d'échec de la mise à jour des objets dans le serveur,
         * restaure la géométrie de l'objet principal et exécute la restauration des objets liés
         */
        const cancelReversedGeometries = () => {
          if (scope.reversingLineData) {
            require('toastr').error($filter('translate')('rulecfg.reverseLineFailed'));
            // restaure le sens d'origine de la géométrie
            scope.editdescription.editedfeature.setGeometry(scope.reversingLineData.backupGeometry);
            // restaure le sens d'origine de la géométrie de chaque objet relié
            restoreRelatedFeaturesGeometries(scope.reversingLineData.relFeatsBackupGeoms,
              scope.editdescription);
          }


        };
      },
    };
  };
  gcelement.$inject = ['EditFactory', 'EditTypesFactory', '$q', 'EditRulesFactory', 'gclayers',
    'gcRestrictionProvider', 'CopyPasteAttributeFactory', 'bizeditProvider', 'FeatureTypeFactory',
    '$timeout', 'FeatureAttachmentFactory', 'gaJsUtils', 'mapJsUtils', 'BizEditFactory',
    '$filter', 'EditDescription'];
  return gcelement;
});
