/**
 *
 */
'use strict';
define(function() {
  var bizeditProvider = function() {
    this.$get = function($filter, gaJsUtils, mapJsUtils, gclayers, FilesFactory, FeatureTypeFactory,
      FeatureAttachmentFactory, $q, $timeout, gcRestrictionProvider) {
      var bizeditProvider = function() {
        this.checkMandatory = (
          scope,
          attributs,
          objectsNo,
          selectedAttributs,
          attributsValues
        ) => {
          scope.missingMandatoryAttributes = this.getMissingMandatoryAttributes(
            attributs, objectsNo, selectedAttributs, attributsValues
          ).map(attribute => attribute.alias);
          if (scope.missingMandatoryAttributes.length) {
            let errorMessage =
              scope.missingMandatoryAttributes.length > 1
                ? 'edition.missing_attributes_message'
                : 'edition.missing_attribute_message';
            require('toastr').error(
              $filter('translate')(errorMessage) +
                scope.missingMandatoryAttributes.join(', ') +
                '.'
            );
            return false;
          }
          return true;
        };

        this.getMissingMandatoryAttributes = (
          attributs, objectsNo, selectedAttributs, attributsValues
        ) => {
          return (attributs || [])
            .filter(attribut => {
              let mandatoryAttribute =
                attribut.mandatory &&
                (objectsNo == 1 || selectedAttributs[attribut.name] == true);
              let attributValue = attributsValues[attribut.name];
              return mandatoryAttribute &&
                (
                  attributValue === undefined ||
                  attributValue === null ||
                  (
                    angular.isString(attributValue) &&
                    attributValue.trim() === ''
                  )
                )
            });
        }

        this.isInvalidValue = function(valueToCheck) {
          return (
            valueToCheck === undefined ||
            (angular.isString(valueToCheck) && valueToCheck.trim() === '')
          );
        };

        this.MARKER_POSITION_LAYER_ID = 'marker-position-layer';

        /**
         * Place le point au centre de la carte.<br>
         * Définit le style du marqueur positionnant le point en cours de création.
         * @param {ol.Map} map carte openlayers de l'application KIS-MAP
         * @param {ol.Feature} marker feature openlayers représentant l'emplacement du point en cours de création
         */
        this.drawMarker = (map, marker) => {
          const iconStyle = new ol.style.Style({
            image: new ol.style.Icon( /** @type {olx.style.IconOptions} */ ({
              anchor: [0.5, 36],
              anchorXUnits: 'fraction',
              anchorYUnits: 'pixels',
              src: 'img/widget/adressLocation/marker.png'
            }))
          });
          marker.setStyle(iconStyle);

          // creation de la layer du marqueur
          const markerPositionLayer = mapJsUtils.addOrGetLayerByProperty(map, 'id', this.MARKER_POSITION_LAYER_ID);

          // créé la source
          const vectorSource = new ol.source.Vector({
            features: [marker]
          });
          markerPositionLayer.setSource(vectorSource);

          // positionne la puce de localisation au-dessus de tous les dessins vectoriels
          const zIndexMax = gaJsUtils.getZIndexMax(map);
          if (zIndexMax) {
            markerPositionLayer.setZIndex(zIndexMax +1);
          }
          // centre la carte sur le marqueur (sans modifier le zoom)
          map.getView().setCenter(marker.getGeometry().getCoordinates());
        };

        /**
         * Enlève la layer openlayers du marqueur de la map après avoir purgé la source de la layer.
         * @param {ol.Map} map carte openlayers de l'application KIS-MAP
         */
        this.removeMarkerLayer = (map) => {
          const markerPositionLayer = mapJsUtils.getLayerByProperty(map, 'id', this.MARKER_POSITION_LAYER_ID);
          if (markerPositionLayer) {
            if (markerPositionLayer.getSource()) {
              markerPositionLayer.getSource().clear();
            }
            map.removeLayer(markerPositionLayer);
          }
        };

        /**
         * Enlève la layer openlayers du parcours réseau après avoir purgé la source de cette layer
         */
        this.removeValidPathLayer = (map) => {
          const validPathLayer = mapJsUtils.getLayerByProperty(map, 'id', 'valid-path-layer');
          if (validPathLayer) {
            validPathLayer.getSource().clear();
            map.removeLayer(validPathLayer);
          }
        };

        /**
         * Vide et retire les layers temporaires du parcours réseau.
         * Utile, par exemple, si l'utilisateur annule la création de point sans avoir terminé un parcours
         * @param {ol.Map} map carte ol de l'application MAP
         */
        this.networkPathLayersCleanRemoval = (map) => {
          const pathLayers = ['valid-path-layer', 'origin-layer', 'paths-layer', 'last-nodes-layer'];
          for (const pathLayer of pathLayers) {
            const mapPathLayer = map.getLayers().getArray().find(layer => layer.get('id') === pathLayer);
            if (mapPathLayer) {
              if (mapPathLayer.getSource()) {
                mapPathLayer.getSource().clear();
              }
              map.removeLayer(mapPathLayer);
            }
          }
        };

        /**
         * Supprime les intéractions du parcours réseau présentes
         * si on annule la création/modiication de point/ligne avant de valider le parcours réseau
         * @param {ol.Map} map carte openlayers de l'application KIS-MAP
         */
        this.networkSelectInteractionRemoval = (map) => {
          const interactionIdToRemove = ['network-path-select', 'network-node-select'];
          const interactionsToRemove = [];
          map.getInteractions().forEach(interaction => {
            if (typeof interaction.get('id') === 'string' && interactionIdToRemove.includes(interaction.get('id'))) {
              interaction.setActive(false);
              interactionsToRemove.push(interaction);
            }
          });
          for (const interaction of interactionsToRemove) {
            map.removeInteraction(interaction);
          }
        };


        const GEOJSON_FORMAT = new ol.format.GeoJSON();

        /**
         * Calcule l'emplacement d'un point sur une géométrie linéaire
         * puis exécute le dessin du marqueur et le centrage de la map sur celui-ci
         * @param {ol.Map} map carte openlayers de l'application KIS-MAP
         * @param {object | ol.geom.MultiLineString | ol.geom.LineString} path géométrie du parcours réseau ol.geom.(Multi)Linestring
         * ou objet du parcours réseau contenant une géométrie linéaire en geojson
         * @param {object} pointByDistance distance depuis le noeud de départ à laquelle doit se trouver le point à calculer
         */
        this.drawMarkerOnPathByDistance = (map,path, pointByDistance) => {

          let pathGeometry = path;

          if (!pathGeometry instanceof ol.geom.MultiLineString || !pathGeometry instanceof ol.geom.LineString) {
            // convertit le parcours dans le sens saisi en géométrie openlayers
            pathGeometry = GEOJSON_FORMAT.readGeometry(path.geometry);
          }

          // calcule le point situé à la distance fournie du noeud de départ du parcours
          if (pathGeometry.getType() === 'LineString') {
            const fraction = pointByDistance.distance / pathGeometry.getLength();
            pointByDistance.coords = pathGeometry.getCoordinateAt(fraction);
          } else {
            pointByDistance.coords = mapJsUtils.getMultiLineStringCoordinatesAt(pathGeometry, null, pointByDistance.distance);
          }

          // évalue si le calcul précédent a réussi
          if (Array.isArray(pointByDistance.coords)) {

            // création du marqueur
            const marker = new ol.Feature({
              geometry: new ol.geom.Point(pointByDistance.coords)
            });

            this.drawMarker(map, marker);

            // modifie le boolean qui signale le calcul du marqueur (active le bouton "Créer le point")
            if (pointByDistance.hasOwnProperty('validators')) {
              pointByDistance.validators.hasPosition = true;
            }

          } else {
            // le calcul du point situé à la distance fournie du noeud de départ du parcours a échoué
            console.error('Erreur lors du calcul les coordonnées de l\'emplacement du point', pointByDistance.coords);
          }
        };

        /**
         * Positionne un marqueur sur le sommet de la ligne en cours de création
         * @param {ol.Map} map carte openlayers de l'application KIS-MAP
         * @param {number[][]|string[][]} lineByCoords tableau des coordonnées de la ligne en cours de création/édition
         * @param {object} selectedsrid srid et description du fti sélectionné
         * @param {number} index rang du sommet parmi les sommets de la ligne
         * @param {string} mapProjDescription description de la projection de la carte
         */
        this.getVertexPosition = (map, lineByCoords, selectedsrid, index, mapProjDescription) => {
          if (Number.isInteger(index) && Array.isArray(lineByCoords)
              && lineByCoords.length > index && Array.isArray(lineByCoords[index])) {

            // détermine les coordonnées du sommet dans la projection de la carte
            const vertexCoordsMapSrid = lineByCoords[index];
            const mapSrid = map.getView().getProjection().getCode();

            // vérifie si un autre projection est sélectionnée
            const hasSelectDistinctSrid = selectedsrid.name && selectedsrid.name.length > 0 && selectedsrid.name !== mapSrid;

            // copie les coordonnées du sommet
            let vertexCoords = vertexCoordsMapSrid.slice();

            if (hasSelectDistinctSrid) {
              // convertit les coordonnées du sommet dans la projection sélectionnée
              try {
                vertexCoords = ol.proj.transform(vertexCoordsMapSrid, selectedsrid.name, mapSrid);
              } catch(e) {
                if (selectedsrid.description && mapProjDescription) {
                  vertexCoords = proj4(selectedsrid.description, mapProjDescription, vertexCoordsMapSrid)
                } else {
                  require('toastr').error($filter('translate')('bizedition.bycoords.unknownProj'));
                }
              }
            }

            // créé le marqueur
            let marker = new ol.Feature({
              geometry: new ol.geom.Point(vertexCoords)
            });

            // affiche le marqueur sur la carte dans une layer 'marker-position-layer'
            this.drawMarker(map, marker);
          }
        };

        /**
         * Créé l'objet ol.Feature de la ligne en cours de création/édition
         * @param {ol.Map} map carte openlayers de l'application KIS-MAP
         * @param {number[][]|string[][]} lineByCoords tableau des coordonnées de la ligne en cours de création/édition
         * @param {object} selectedsrid srid et description du fti sélectionné
         * @param {string} mapProjDescription description de la projection de la carte
         * @return {ol.Feature} objet openlayers de la ligne en cours de création/édition de géométrie MultiLineString
         */
        this.createDraftLine = (map, lineByCoords, selectedsrid, mapProjDescription) => {
          const draftLineByCoords = new ol.Feature();
          const mapSrid = map.getView().getProjection().getCode();
          const hasSelectedDistinctSrid = selectedsrid.name && selectedsrid.name.length > 0 && selectedsrid.name !== mapSrid;
          let coords = lineByCoords;
          if (hasSelectedDistinctSrid && selectedsrid.description && mapProjDescription) {
            // convertit les coordonnées du sommet dans la projection sélectionnée
            coords = lineByCoords.map(coords => proj4(selectedsrid.description, mapProjDescription, coords));
          }

          const draftLineGeom = new ol.geom.MultiLineString([coords]);
          draftLineByCoords.setGeometry(draftLineGeom);
          const draftLineStyle = new ol.style.Style({
            stroke: new ol.style.Stroke({
              width: 10,
              color: 'rgba(26, 140, 255, 0.7)'
            })
          });
          draftLineByCoords.setStyle(draftLineStyle);
          return draftLineByCoords;
        };

        // layer openlayers du tracé temporaire de la ligne en cours de création
        let draftLineByCoordsLayer;
        const DRAFT_LINE_COORD_ID = 'draft-line-coords-layer';

        /**
         * Exécuté:<ul><li>
         * à chaque blur d'un input de coordonnées dans la popup "Créer une ligne par coordonnées".</li><li>
         * à l'ouverture de la popup "Créer une ligne par coordonnées" après sélection d'une ligne existante à modifier</li><li>
         * après suppression d'un sommet de ligne</li></ul>
         * Affiche provisoirement la ligne en cours de création si le tableau de coordonnées a une longueur supérieure à 1.
         * Si la longueur du tableau de coordonnées de la directive est inférieure à 2 alors la couche du tracé est enlevée de la carte.
         * @param {ol.Map} map carte openlayers de l'application KIS-MAP
         * @param {number[][]} lineByCoords tableau de coordonnées de la ligne
         * @param {object} selectedsrid objet contenant le code et la description de la projection
         * @param {number|null} index rang du sommet de ligne dont on modifie les coordonnées XY.
         * @param {boolean} hasDeletedVertex est true pour signaler la suppression d'un sommet de ligne
         * @param {boolean} isStartingEditLine est true quand la méthode est exécutée à l'ouverture de la popup "Edition d'une ligne par coordonnées" après sélection d'un objet ligne
         * @param {string} mapProjDescription description de la projection de la carte
         */
        this.buildDraftLineByCoords = (map, lineByCoords, selectedsrid,
            index, hasDeletedVertex, isStartingEditLine, mapProjDescription) => {

          if (Array.isArray(lineByCoords) && lineByCoords.length > 1) {

            // récupère les coordonnées du point ajouté et vérifie le type nombre des coordonnées saisies
            const editedCoords = index ? lineByCoords[index] : null;
            const isCoordsNumber = Array.isArray(editedCoords) &&  editedCoords.length === 2 ?
              Number.isFinite(editedCoords[0]) && Number.isFinite(editedCoords[1]) : false;

            // construit la ligne si le dernier sommet ajouté est bien saisi
            // construit la ligne à l'ouverture de l'édition d'une ligne existante
            // rafraîchit la ligne après suppression d'un sommet
            if (isCoordsNumber || hasDeletedVertex || isStartingEditLine) {

              // créé la feature openlayers à la géométrie de type MultiLineString
              const draftLineByCoords = this.createDraftLine(map, lineByCoords, selectedsrid, mapProjDescription);

              // créé la source
              const vectorSource = new ol.source.Vector({
                features: [draftLineByCoords]
              });

              if (!draftLineByCoordsLayer) {
                // initialise la layer du tracé temporaire de la ligne si elle n'existe pas
                draftLineByCoordsLayer = new ol.layer.Vector();
                draftLineByCoordsLayer.setZIndex(10004);
                draftLineByCoordsLayer.set('id', DRAFT_LINE_COORD_ID);
              } else {
                // ré-initialise la source de la layer existant déjà
                draftLineByCoordsLayer.getSource().clear();
              }
              draftLineByCoordsLayer.setSource(vectorSource);

              // insère la layer dans la map si elle en est absente
              if (!mapJsUtils.mapHasLayerByProperty(map, 'id', DRAFT_LINE_COORD_ID)) {
                map.addLayer(draftLineByCoordsLayer);
              }
            }
          } else {
            // efface la ligne quand il reste moins de 2 sommets
            this.draftLineByCoordsCleanRemoval(map);
          }
        };

        /**
         * Vide et enlève de la carte la couche du tracé provisoire de la ligne en cours de construction depuis la popup "Créer une ligne par coordonnées".
         * Exécuté si la ligne n'a plus qu'un seul sommet ou bien à l'annulation/fermeture de la popup.
         * @param {ol.Map} map carte openlayers de l'application KIS-MAP
         */
        this.draftLineByCoordsCleanRemoval = (map) => {
          if (mapJsUtils.mapHasLayerByProperty(map,'id', DRAFT_LINE_COORD_ID) && draftLineByCoordsLayer) {
            if (draftLineByCoordsLayer.getSource()) {
              draftLineByCoordsLayer.getSource().clear();
            }
            map.removeLayer(draftLineByCoordsLayer);
          }
        };

        /**
         * Dans la popup "Création/Edition d'une ligne par coordonnées".
         * Teste si deux points valides sont saisis pour activer/désactiver le bouton "Créer/modifier la ligne"
         * @return {boolean} true si la liste de coordonnées contient au moins 2 éléments et ne contient que des nombres (en string ou pas)
         */
        this.hasTwoCoordinatesValid = (lineByCoords) => {
          const hasCoordsValid = (coords) => {
            let isValid = false;
            if (Array.isArray(coords)) {
              for (const coord of coords) {
                isValid = Number.isFinite(coord);
                if (!isValid) {
                  return isValid;
                }
              }
            }
            return isValid;
          };
          if (Array.isArray(lineByCoords) && lineByCoords.length > 1) {
            return lineByCoords.every(hasCoordsValid);
          }
          return false;
        };

        /**
         * Au clic sur le bouton "Enregistrer" dans la popup de création d'une ligne par positionnement linéaire, créé la géométrie de la ligne par positionnement linéaire
         * @param {ol.geom.LineString} lastSegment dernière ligne simple de la géométrie MultiLineString du parcours réseau
         * @param {object} lineByDistance objet contenant les coordonnées du point de base, la longueur du branchement,
         * l'angle avec le collecteur et le côté du collecteur où le branchement doit être dessiné
         * @return {ol.geom.LineString}
         */
        this.createConnectionByAngleAndDistance = (lastSegment, lineByDistance) => {
          const endConnectionCoords = lineByDistance.coords;
          const distance = lineByDistance.length;
          const angle = lineByDistance.angle;
          const side = lineByDistance.side;
          let connectionGeometry;
          const lastSegmentGisement = mapJsUtils.calculateGisement(lastSegment);
          const angleBySide = (side === 'left' || !side) ? angle : (0 - angle);
          let connectionGisement = lastSegmentGisement + 180 + angleBySide;
          if (connectionGisement < 0) {
            connectionGisement = connectionGisement + 360;
          }
          if (connectionGisement > 360) {
            connectionGisement = connectionGisement - 360;
          }
          const startConnectionCoords = mapJsUtils.calculateCoodinatesByGisementAndDistance(endConnectionCoords, connectionGisement, distance);
          if (Array.isArray(startConnectionCoords)) {
            connectionGeometry = new ol.geom.LineString([startConnectionCoords, endConnectionCoords]);
          }
          return connectionGeometry;
        };

        this.CONNECTION_NODE_STYLE = new ol.style.Style({
          image: new ol.style.Circle({
            radius: 6,
            fill: new ol.style.Fill({
              color: 'rgba(255,255,255,1)',
            }),
            stroke: new ol.style.Stroke({
              color: 'rgba(247,26,255,0.7)',
            }),
          }),
        });
        this.CONNECTION_PATH_STYLE = new ol.style.Style({
          stroke: new ol.style.Stroke({
            width: 10,
            color: 'rgba(247,26,255,0.3)'
          })
        });

        this.CONNECTION_LAYER_ID = 'connection-layer';

        /**
         * Dessine une ligne par positionnement linéaire : <ul><li>
         *   Calcule le point de raccord de la ligne sur le tronçon</li><li>
         *   Calcule l'orientation du segment de tronçon portant le point de raccord</li><li>
         *   Calcule le point de départ de la ligne en connaissant l'angle et la distance depuis le point de raccord</li><li>
         *   Dessine la ligne sur la carte (insère dans une couche <i>connection-layer</i> qui existe déjà ou que l'on créé ici)</li></ul>
         * @param {ol.Map} map carte openlayers de l'application MAP
         * @param {object} networkPath conteneur de la ligne du parcours réseau dans le sens défini par l'utilisateur (ne tient pas compte du sens de digitalisation du tronçon)
         * @param {object} lineByDistance objet contenant les données de la popup saisies par l'utilisateur (angle, distance, côté et longueur)
         */
        this.calculateAndDrawConnection = (map, networkPath, lineByDistance) => {
          // récupère le parcours réseau dans le sens de saisie par l'utilisateur (avec des tronçons éventuellement saisis dans le sens inverse de digit)
          const pathGeometry = networkPath.fullData.userDirectionPath;

          // calcule les coordonnées du point de raccord de la ligne sur le tronçon
          // isole le dernier tronçon du parcours
          let lastLineString;
          if (pathGeometry.getType() === 'LineString') {
            const fraction = lineByDistance.distance / pathGeometry.getLength();
            lineByDistance.coords = pathGeometry.getCoordinateAt(fraction);
            lastLineString = pathGeometry;
          } else {
            lineByDistance.coords = mapJsUtils.getMultiLineStringCoordinatesAt(pathGeometry, null, lineByDistance.distance);
            lastLineString = pathGeometry.getLineStrings()[pathGeometry.getLineStrings().length -1];
          }

          if (Array.isArray(lineByDistance.coords)) {

            let lastSegment;
            const lastLineStringCoords = lastLineString.getCoordinates();

            // évalue le nombre de sommets de la ligne
            // si nbSommets > 2 alors on doit récupérer le segment portant le point de raccord pour le calcul de gisement
            if (lastLineStringCoords.length > 2) {
              const connectionPoint = new ol.geom.Point(lineByDistance.coords);

              // la ligne a plus de 2 sommets
              // isole le segment de ligne portant le point de raccord
              for (let i = 0; i < lastLineStringCoords.length - 1; i++) {
                const lineSegment = new ol.geom.LineString([lastLineStringCoords[i], lastLineStringCoords[i + 1]]);
                if (lineSegment.intersectsExtent(connectionPoint.getExtent())) {
                  lastSegment = lineSegment;
                  break;
                }
              }
            } else {
              // la ligne n'a que 2 sommets, on récupère celle-ci directement pour le calcul de gisement
              lastSegment = lastLineString;
            }

            // calcule la géométrie de la ligne par la méthode gisement/distance
            const connectionGeometry = this.createConnectionByAngleAndDistance(lastSegment, lineByDistance);
            if (!connectionGeometry) {
              console.error('Erreur lors du calcul de la géométrie de la ligne', lineByDistance);
              return;
            }

            // créé la ligne ('connection') et les noeuds (départ et raccord)
            const connection = new ol.Feature({
              geometry: connectionGeometry
            });
            connection.set('id', 'connection-feature');
            connection.setStyle(this.CONNECTION_PATH_STYLE);

            const startNode = new ol.Feature({
              geometry: new ol.geom.Point(connectionGeometry.getFirstCoordinate())
            });
            const endNode = new ol.Feature({
              geometry: new ol.geom.Point(connectionGeometry.getLastCoordinate())
            });

            startNode.setStyle(this.CONNECTION_NODE_STYLE);
            endNode.setStyle(this.CONNECTION_NODE_STYLE);

            // dé-sélectionne la ligne source
            if (gclayers.getselectSource()) {
              gclayers.getselectSource().clear();
            }

            let connectionLayer = mapJsUtils.getLayerByProperty(map, 'id', this.CONNECTION_LAYER_ID);
            const source = new ol.source.Vector({
              features: [connection, startNode, endNode]
            });

            // insère la ligne calculée dans une couche openlayers ajoutée à la carte
            if (!connectionLayer) {
              connectionLayer = new ol.layer.Vector();
              connectionLayer.set('id', this.CONNECTION_LAYER_ID);
              connectionLayer.setSource(source);
              connectionLayer.setZIndex(10006);
              map.addLayer(connectionLayer);
            } else {
              connectionLayer.setSource(source);
            }

            // active le bouton "Enregistrer" de la popup
            lineByDistance.validators.hasConnection = true;

            // centre la carte sur la ligne créée
            map.getView().setCenter(connectionGeometry.getCoordinateAt(0.5));
          } else {
            console.error('Erreur lors du calcul des coordonnées du raccord sur le tronçon : ', lineByDistance.coords);
          }
        };

        /**
         * Purge la layer temporaire de la création/édition de ligne et retire cette layer de la carte
         * @param {ol.Map} map carte openlayers de l'application MAP
         */
        this.removeConnectionLayer = (map) => {
          const connectionLayer = mapJsUtils.getLayerByProperty(map, 'id', this.CONNECTION_LAYER_ID);
          if (connectionLayer) {
            if (connectionLayer.getSource()) {
              connectionLayer.getSource().clear();
            }
            map.removeLayer(connectionLayer);
          }
        };

        /**
         * Récupère le nom du répertoire de chaque attribut de type attachment de l'objet dont les attributs sont en cours d'édition
         * @param {object} jsonFeature objet créé renvoyé du backend et parsé en json
         * @param {string[]} attributes liste des attributs de type attachment(s) saisis dans la popup
         * @param {number} index rang de l'attribut à traiter dans le tableau des attributs
         * @param {object[]} allFields tableau contenant toutes les configurations des champs de la popup
         * @param {object} attachmentMapping contient les noms de sous-dossiers des fichiers attachés de chaque attribut de l'objet
         * @param defer
         * @return {Promise} contenant les noms de sous-dossiers des fichiers attachés de chaque attribut de l'objet
         */
        this.findFeatureAttachmentsProcessId = (jsonFeature, attributes, index, allFields, attachmentMapping, defer) => {
          if (!defer) {
            defer = $q.defer();
          }
          const attributeName = attributes[index];
          // recherche le nom du sous-dossier upload dans le field de l'attribut (il peut avoir plusieurs objets ayant le même attribut)
          const attributeFields = allFields.filter(field => field.name === attributeName && field.hasOwnProperty('attachmentId'));
          if (!attachmentMapping) {
            attachmentMapping = {};
          }
          this.findAttributeAttachmentsField(jsonFeature, attributeFields, 0, attachmentMapping).then(
            field => {
              if (typeof field === 'object' && field !== null) {
                attachmentMapping[field.name] = field.attachmentId;
              } else {
                console.error(
                  'copyFilesFromUploadToAttachmentFolder : Impossible de localiser les fichier attachés de l\'attribut '
                      + attributeName + ' pour l\'objet ', jsonFeature);
              }
              if (++index < attributes.length) {
                return this.findFeatureAttachmentsProcessId(jsonFeature, attributes, index, allFields, attachmentMapping, defer);
              } else {
                defer.resolve(attachmentMapping);
              }
            },
            () => {
              if (++index < attributes.length) {
                return this.findFeatureAttachmentsProcessId(jsonFeature, attributes, index, allFields, defer);
              } else {
                defer.resolve(attachmentMapping);
              }
            }
          );
          return defer.promise;
        };

        /**
         * Recherche le nom du sous-dossier où sont situés le(s) fichier(s) attaché(s) de l'attribut de l'objet.
         * Le nom du sous-dossier est contenu dans la propriété attachmentId du champ de l'attribut.
         * Il n'y a pas de liaison évidente entre les fichiers et l'objet auquel ils doivent être rattachés
         * car l'objet ne possèdait pas d'id quand les fichiers ont été uploadés.
         * On regarde toutes les propriétés "attachmentId" issue de la configuration des champs des attributs
         * puis on évalue si les fichiers du sous-dossier portant le nom "attachmentId" sont identiques aux fichiers définis dans l'attribut de l'objet
         * @param {object} jsonFeature
         * @param {object[]} attributeFields propriétés fieldData des attributs de la popup (tous les onglets réunis)
         * @param {number} index rang de l'attribut de type attachment à évaluer dans le tableau attributeFields. Cette gestion permet de parcourir 1 après l'autre les attributs.
         * @param {object} existingMapping objet contenant les attachmentId déjà traités
         * @param defer
         * @return {Promise}
         */
        this.findAttributeAttachmentsField = (jsonFeature, attributeFields, index, existingMapping, defer) => {
          if (!defer) {
            defer = $q.defer();
          }
          const field = attributeFields[index];
          FilesFactory.listUploadFiles(field.attachmentId).then(
            res => {
              if (res.data && Array.isArray(res.data)) {
                const attachmentFiles = res.data;
                const attachmentFeature = jsonFeature.properties[field.name].split(',');

                const isFilesCountEquals = attachmentFiles.length === attachmentFeature.length;
                const isFeatureFilesAndRepoFilesEquals = attachmentFeature.every(attachName => attachmentFiles.includes(attachName));
                const isFilesAlreadyProcessed = Object.values(existingMapping).includes(field.attachmentId);

                if (isFilesCountEquals && isFeatureFilesAndRepoFilesEquals && !isFilesAlreadyProcessed) {
                  defer.resolve(field);
                } else {
                  if (++index < attributeFields.length) {
                    return this.findAttributeAttachmentsField(jsonFeature, attributeFields, index, existingMapping, defer);
                  } else {
                    defer.resolve(null);
                  }
                }
              } else {
                defer.resolve(null);
              }
            }
          ).catch(()=> {
            defer.resolve(null);
          });
          return defer.promise;
        };

        /**
         * Copie les fichiers attachés de chaque objet créé du dossier UPLOAD vers le dossier ATTACHNMENTS.
         * On copie tous les fichiers d'un objet avant de passer à l'objet suivant.
         * @param {object[]} featsCreated tableau des objets créés avec id renvoyés par le backend
         * @param {number} index rang dans le tableau <code>featsCreated</code> de la feature à traiter.
         * @param {object} fieldDatas configuration des champs d'attributs de la popup
         */
        this.copyFilesFromUploadToAttachmentFolder = (featsCreated, index, fieldDatas) => {
          let jsonFeature;
          try {
            jsonFeature = JSON.parse(featsCreated[index].json);
          } catch(e) {
            require('toastr').error($filter('translate')('featureattachment.createdFeatureCorrupted'));
            console.error(e);
            return;
          }

          // fti (clé des objets contenant les objets des fichiers attachés de chaque objet)
          const fti = this.getFtiFromJsonFeature(jsonFeature, featsCreated, index);
          const attachmentTypes = ['g2c.attachment','g2c.attachments'];

          // filtre les attributs de type attachment présents dans la feature
          const attachmentAttributes = fti.attributes
            .filter(attr => attachmentTypes.includes(attr.type)).map(attr => attr.name)
            .filter(attrName => Object.keys(jsonFeature.properties).includes(attrName));


          // Isole les fieldData des champs de la popup
          // attention, les fieldDatas ne sont pas structurés de la même façon en présence de fiches objet
          const allFields = this.getAttributePopupFieldDatas(fti, fieldDatas);

          if (Array.isArray(allFields) && allFields.length > 0) {
            if (attachmentAttributes.length > 0) {
              this.findFeatureAttachmentsProcessId(jsonFeature, attachmentAttributes, 0, allFields, null).then(
                attachmentMap => {
                  if (typeof attachmentMap === 'object' && attachmentMap !== null) {
                    const copyFeatureAttachmentPromises = [];

                    for (const [attributeName, attachmentId] of Object.entries(attachmentMap)) {
                      const copyAttributeAttachmentPromise = FeatureAttachmentFactory.attachImportedFiles(
                        attachmentId, fti.uid, [attributeName], [featsCreated[index]]);
                      copyFeatureAttachmentPromises.push(copyAttributeAttachmentPromise);
                    }

                    $q.all(copyFeatureAttachmentPromises).then(
                      res => {
                        console.info('retour de l\'attachement des fichiers : ', res);
                        if (++index < featsCreated.length) {
                          this.copyFilesFromUploadToAttachmentFolder(featsCreated, index, fieldDatas);
                        } else {
                          require('toastr').success($filter('translate')('featureattachment.attachmentSuccess'));
                        }
                      },
                      err => {
                        console.error(err);
                        if (++index < featsCreated.length) {
                          this.copyFilesFromUploadToAttachmentFolder(featsCreated, index, fieldDatas);
                        }
                      }
                    );
                  }
                }
              );
            } else  if (++index < featsCreated.length) {
              this.copyFilesFromUploadToAttachmentFolder(featsCreated, index, fieldDatas);
            }
          } else {
            console.error('copyFilesFromUploadToAttachmentFolder : les propriétés fieldData des '
                + 'composants formField de la popup attribute_popup2 ont subi une régression '
                + 'empêchant la récupération de l\'emplacement des fichiers attachés à copier. '
                + 'Veuillez vérfier la méthode getFieldDatas '
                + 'et la construction des fieldDatas de la popup dans bizeditwidget#createAttributePopupFieldDatas');
          }
        };

        /**
         * Construit un tableau contenant la configuration des champs de la popup (i.e fieldData).
         * Rassemble la configuration des champs de tous les features nouvellement créés pour le composant défini en paramètre.
         * Cette méthode est adaptée aussi bien à la structure classique qu'à la structure par onglets des fiches-objet.
         * @param {object} fti featureTypeInfo du composant dont font partie les objets nouvellement créés
         * @param {object} fieldDatas configuration des champs d'attributs de la popup
         * @return {object[]} tableau des propriétés <code>fieldData</code> de tous les <code>formFields</code> de la popup <code>attributePopup2</code>
         */
        this.getAttributePopupFieldDatas = (fti, fieldDatas) => {
          let allFields = [];
          const fieldDatasExist = fieldDatas && fieldDatas[fti.uid] && Object.keys(fieldDatas[fti.uid]).length > 0;
          if (fieldDatasExist) {

            // Si les fieldDatas du 1er feature du fti sont organisés en tableau alors il s'agit d'une structure classique.

            // Si les fieldDatas du 1er feature du fti contiennent des objets (correspondant aux onglets) dans lesquels sont répartis les fields
            // alors il s'agit d'une structure de fiche-objet
            const isFicheObjetConfig = !Array.isArray(Object.values(fieldDatas[fti.uid])[0]);

            const allFeaturesFields = Object.values(fieldDatas[fti.uid]);

            if (isFicheObjetConfig) {
              for (const featureFields of allFeaturesFields) {
                for (const tabFields of Object.values(featureFields)) {
                  if (Array.isArray(tabFields)) {
                    allFields.push(...tabFields);
                  }
                }
              }
            } else {
              for (const featureFields of allFeaturesFields) {
                if (Array.isArray(featureFields)) {
                  allFields.push(...featureFields);
                }
              }
            }
          }
          return allFields;
        };

        /**
         * Modifie les hauteurs d'un corps d'onglet en fonction du nombre de lignes de titres d'onglets.<br>
         * Exemple: 1 ligne d'onglets (40px) et tab-content (800px) => 2 lignes d'onglets (80px) et tab-content (760px).<br>
         * Méthode utilisée pour modifier la hauteur du corps des onglets de la fiche-objet et de l'intervention simple
         * @param {string} iddiv attribut id de l'élément HTML de la popup
         * @param {string} ruleIdentifier chaîne permettant d'identifier la règle CSS à modifier dans le fichier "g2c_template.css"
         * @param {string} tabContainerCssClass classe css du conteneur des onglets
         * @param {number} triesLimit nombre de tentatives
         */
        this.adjustObjectFileTabContentHeight = (iddiv, ruleIdentifier, tabContainerCssClass, triesLimit) => {
          if (iddiv) {
            // recherche l'élément principal de la popup
            const mainContainer = document.getElementById(iddiv);
            if (mainContainer) {

              // recherche le conteneur des onglets (titre + corps)
              const tabContainer = mainContainer.querySelector('.' + tabContainerCssClass);
              if (tabContainer) {

                // recherche la barre des onglets
                const ul = tabContainer.querySelector('.nav.nav-tabs');
                if (ul) {

                  // calcule le nombre de lignes d'onglets
                  const tabLinesCount = Math.round(ul.clientHeight / 40);
                  if (tabLinesCount > 1) {

                    // calcule la hauteur à enlever au corps de l'onglet (40px par ligne d'onglet supplémentaire)
                    const exceedingTabHeight = (tabLinesCount - 1) * 40;

                    // recherche le fichier css "g2c_template.css"
                    const styleSheet = Object.values(Object(document.styleSheets))
                      .find(sheet => sheet.href !== null && sheet.href.endsWith('g2c_template.css'));

                    if (styleSheet) {
                      // recherche la règle CSS du conteneur d'onglets ("tab-content")
                      const cssRules = Object.values(styleSheet.cssRules);
                      const tabRule = cssRules.find(rule => typeof rule.selectorText === 'string'
                          && rule.selectorText.includes(ruleIdentifier));
                      if (tabRule) {

                        // modifie la propriété déclaration "height" de la règle
                        tabRule.style.height = 'calc(100% - ' + (41 + exceedingTabHeight) + 'px)';
                      }
                    }
                  }
                }
              } else if (triesLimit >= 0) {
                $timeout(() => {
                  this.adjustObjectFileTabContentHeight(iddiv, ruleIdentifier, tabContainerCssClass, --triesLimit);
                });
              }
            } else if (triesLimit > 0) {
              // attend la prochaine frame soit le temps nécessaire pour que la popup puissent apparaitre
              $timeout(() => {
                this.adjustObjectFileTabContentHeight(iddiv, ruleIdentifier, tabContainerCssClass, --triesLimit);
              });
            }
          }
        };

        /**
         * Appelée une fois le premier point de la forme saisi,
         * cette méthode, fabrique la géoémtrie qui sera correcte
         * pour que le service adéquat puisse vérifier que l'on travaille
         * dans la zone de restriction.
         *
         * @param {*} feature Paramétre en entrée/sortie.
         *                    Feature Openlayers qui contient
         *                    la géométrie à vérifier
         * @returns Sans Objet
         */
        const checkfeature = (feature) => {
          let coordinates;
          if (feature.geometry.type === 'Point') {
            //-- Pas besoin de modifier la géoémtrie du point.
            return;
          }
          //-- Faire nen sorte que coordinates contienne directement
          //-- la liste de coordonnées, et non une liste de liste
          //-- ou même une liste de liste de liste.
          coordinates = feature.geometry.coordinates;
          while (Array.isArray(coordinates[0][0])) {
            coordinates = feature.geometry.coordinates[0];
          }
          if (coordinates.length === 1) {
            coordinates[0].push([coordinates[0][0], coordinates[0][1]]);
          }
          if (coordinates.length === 2
            && coordinates[0][0] === coordinates[1][0]
            && coordinates[0][1] === coordinates[1][1]
          ) {
            //-- Une ligne avec 2 points identiques ne pourra pas être testée
            //-- correctement, or, quand on saisi le premier point d'une forme,
            //-- c'est ce que l'on obtient ...
            coordinates[1][0] += 0.1;
            if (feature.geometry.type === 'Polygon') {
              feature.geometry.type = 'LineString';
            }
          }
        };


        /**
         * Vérifie que l'on va créer (au moins le premier point) la forme
         * dans la zone de restriction géographqiue quand il y en a une.
         *
         * @param {*} evtFeature Feature de l'événement OpenLayers
         *     (suite à clic pour positionnement du premier point de la forme)
         * @param {*} map  Carte OpenLayers
         * @returns
         */
        this.checkEditedFeature = (evtFeature, map) => {
          const defer = $q.defer();
          const format = new ol.format.GeoJSON();
          const feature = format.writeFeatureObject(evtFeature);

          if (!gcRestrictionProvider.hasRestrictionEdition()) {
            //-- Pas de restriction donc c'est OK
            defer.resolve({ restrictionOk: true });
          }
          else {
            //-- Restriction, donc vérifier
            checkfeature(feature);
            gcRestrictionProvider.GeometryInRestriction(
              feature.geometry, map.getView().getProjection().getCode()
            ).then(
              (res) => {
                if (!JSON.parse(res.data)) {
                  gcRestrictionProvider.showErrorMessage();
                  defer.resolve({ restrictionOk: false });
                }
                else {
                  defer.resolve({ restrictionOk: true });
                }
              },
              (res) => {
                console.error(res.data);
                defer.resolve({ restrictionOk: true });
              }
            );
          }
          return defer.promise;
        };

        /**
         * Retourne un objet contenant 2 propriétés: nom du composant, id de l'objet.
         * Cet objet est le format requis pour les 2 premiers paramètres de la requête
         * FeatureAttachementFactory.copyFeatureAttachement
         * @param {object} savedFeature objet json dont on extrait le nom du composant et l'id
         * @return {Object} {layername, featureId}
         */
        this.buildRequestParameter = (savedFeature, fti) => {
          if (savedFeature && savedFeature instanceof ol.Feature) {
            const format = new ol.format.GeoJSON();
            savedFeature = JSON.parse(format.writeFeature(savedFeature));
          }
          const id
            = gaJsUtils.getIdInCaseEsriId(savedFeature, fti);
          const sourceIdSeparatorIndex = id.lastIndexOf('.');
          if (sourceIdSeparatorIndex !== -1) {
            const sourceLayerName = id.substring(0, sourceIdSeparatorIndex);
            const sourceId = id;
            return Object({layername: sourceLayerName, featureid: sourceId});
          } else {
            console.error(
              'Impossible d\'initialiser la copie des fichiers attachés pour l\'objet ',
              savedFeature, '. L\'id de l\'objet est mal formé');
            require('toastr').error($filter('translate')('bizedition.formatIdError'));
            require('toastr').error($filter('translate')('bizedition.attachmentError'));
          }
        };

        /**
         * Récupère le fti à partir de l'id d'un objet nouvellement créé/mis à jour
         * @param {object} jsonFeature propriété json de l'objet créé converti en objet
         * @param {object[]} featsCreated tableau des objets créés/mis à jour
         * @param {number} index rang de l'objet dans le tableau des objets créé/mis à jour du composant
         * @return {null|*} fti de l'objet créé ou mis à jour (ou null si erreur)
         */
        this.getFtiFromJsonFeature = (jsonFeature, featsCreated, index) =>  {
          // nom du fti (sert à trouver le dossier du composant dans le dossier ATTACHMENTS)
          let ftiName;
          if (jsonFeature) {
            ftiName = jsonFeature.id.split('.').shift();
          } else if (featsCreated[index] && gaJsUtils.notNullAndDefined(featsCreated[index].id)) {
            // cas esri: feat.json est null, seul l'id est renvoyé du backend jusqu'à présent
            ftiName = featsCreated[index].id.split('.').shift();
          }

          if (!ftiName) {
            require('toastr').error($filter('translate')('featureattachment.createdFeatureWithoutId'));
            console.error($filter('translate')('featureattachment.createdFeatureWithoutId'),
              ' \n Détail: objet créé sans id dans le backend', featsCreated[index]);
            return null;
          }

          // fti (clé des objets contenant les objets des fichiers attachés de chaque objet)
          return FeatureTypeFactory.getFeatureByName(ftiName);
        };

        /**
         *Dès que le bouton d’exécution des règles devient actif
         * alors il devient clignotant jusqu'à ce que l’utilisateur ait cliqué dessus.
         * <a href="https://altereo-informatique.atlassian.net/browse/KIS-3319">KIS-3319 [DEA - Edition] : Amélioration lors de la Mise à jour géométrique</a>
         */
        this.startApplyRulesButtonBlinking = () => {
          const className = 'btn-default-blink-infinite';
          const button = document.getElementById('onModifyEnd');
          if (button && !button.classList.contains(className)) {
            button.classList.add(className);
          }
        };

        /**
         * Au clic sur le bouton "Exécuter les règles onEnd" ou sur le bouton "Réinitialiser"
         * alors le clignotement du bouton "Exécuter les règles onEnd" s'arrête.
         * <a href="https://altereo-informatique.atlassian.net/browse/KIS-3319">KIS-3319 [DEA - Edition] : Amélioration lors de la Mise à jour géométrique</a>
         */
        this.stopApplyRulesButtonBlinking  = (shouldHideButton = false) => {
          const className = 'btn-default-blink-infinite';
          const button = document.getElementById('onModifyEnd');
          if (button) {
            if (button.classList.contains(className)) {
              button.classList.remove(className);
            }
            if(shouldHideButton && !button.classList.contains('hidden')) {
              button.classList.add('hidden');
            }
          }
        };
        /**
         * bizeditmultiupdate<br>
         *Dès que le bouton d’exécution des règles devient actif
         * alors il devient clignotant jusqu'à ce que l’utilisateur ait cliqué dessus.
         * <a href="https://altereo-informatique.atlassian.net/browse/KIS-3319">KIS-3319 [DEA - Edition] : Amélioration lors de la Mise à jour géométrique</a>
         */
        this.toggleApplyRulesButtonBlinking = () => {
          const className = 'btn-default-blink-infinite';
          const button = document.getElementById('onModifyEnd');
          if (button) {
            if (button.classList.contains(className)) {
              button.classList.remove(className);
            } else {
              button.classList.add(className);
            }
          }
        };
      };
      return new bizeditProvider();
    };
    this.$get.$inject = ['$filter', 'gaJsUtils', 'mapJsUtils', 'gclayers', 'FilesFactory',
      'FeatureTypeFactory', 'FeatureAttachmentFactory', '$q', '$timeout', 'gcRestrictionProvider'];
  };

  return bizeditProvider;
});
